<?php
namespace BricksUltimate\ElementConditions;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use BricksUltimate\ElementConditions\Base;

class WooConditions extends Base {

	public function add_element_condition_group( $groups ) {
		$groups[] = [
			'name'  => 'buwooprd',
			'label' => esc_html__( 'Product - BU', 'bricksultimate' ),
		];

		$groups[] = [
			'name'  => 'buwoo',
			'label' => esc_html__( 'WooCommerce - BU', 'bricksultimate' ),
		];

		$groups[] = [
			'name'  => 'buwcpn',
			'label' => esc_html__( 'Woo Coupons - BU', 'bricksultimate' ),
		];

		$groups[] = [
			'name'  => 'buwcord',
			'label' => esc_html__( 'Woo Orders - BU', 'bricksultimate' ),
		];

		return $groups;
	}

	public function add_element_condition( $options ) {
		$options[] = self::getEqualValueTextCondition( 'product_sku', 'Sku', 'buwooprd', '' );
		$options[] = [
			'key'   => 'stock_quantity',
			'label' => esc_html__('Stock quantity', 'bricksultimate'),
			'group' => 'buwooprd',
			'compare' => [
				'type'        => 'select',
				'options'     =>  [
					'==' => '==',
					'!=' => '!=',
					'>=' => '>=',
					'<=' => '<=',
					'>'  => '>',
					'<'  => '<',
				],
				'placeholder' => esc_html__( '==', 'bricks' ),
			],
			'value'   => [
				'type' 	=> 'number',
				'unitless' => true,
				'hasDynamicData' => false,
				'small' => false,
				'medium' => true,
				'placeholder' => 1,
			]
		];

		foreach( $this->get_woo_conditions() as $key => $tag ) {
			$placeholder = $tag['placeholder'] ?? '';
			$info = $tag['info'] ?? '';

			if( isset( $tag['type'] ) && $tag['type'] == 'truefalsetext' ) {
				$options[] = self::getTrueFalseTextCondition( $key, $tag['label'], $tag['group'], $placeholder, $info);
			} elseif( isset( $tag['type'] ) && $tag['type'] == "dropdown2nd" ) {
				$options[] = self::getDropDownCondition2nd($key, $tag['label'], $tag['group'], esc_attr( $placeholder ), $tag['options'] );
			} elseif( isset( $tag['type'] ) && $tag['type'] == "dropdown" ) {
				$options[] = self::getDropDownCondition($key, $tag['label'], $tag['group'], esc_attr( $placeholder ), $tag['options'] );
			} elseif( isset( $tag['type'] ) && $tag['type'] == "math" ) {
				$options[] = self::getMathCondition($key, $tag['label'], $tag['group'], esc_attr( $placeholder ), esc_attr( $info ) );
			} else {
				$options[] = self::getTrueFalseCondition( $key, $tag['label'], $tag['group'] );
			}
		}

		return $options;
	}

	public function result_element_condition( $condition_met, $condition_key, $condition ) {
		$object = \Bricks\Query::get_loop_object();
		$post_id = is_object( $object ) ? $object->ID : get_the_ID();
		$product = false;
		if( ! empty( $post_id ) ) {
			$product = wc_get_product( $post_id );
		}

		switch( $condition_key ) {
			case 'product_in_cart':
				if( ! is_null( WC()->cart ) && isset( $condition['value'] ) ) {
					$ids = bricks_render_dynamic_data( $condition['value'] );
					$ids = strstr( $ids, ',' ) ? explode( ",", $ids ) : [ $ids ];
					$value = false;
					foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) {
						if ( in_array( $cart_item['product_id'], $ids ) ) {
							$value = true;
							break;
						}
					}
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;
			case 'is_shop_page':
				$value = is_shop();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'is_single_product_page': 
				$value = is_product();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'is_cart_page': 
				$value = is_cart();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'is_checkout_page': 
				$value = is_checkout();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'order_pay_page': 
				$value = is_checkout_pay_page();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'is_account_page': 
				$value = is_account_page();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'thank_you_page': 
				$value = is_wc_endpoint_url( 'order-received' );
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'is_cart_empty': 
				$value = WC()->cart->is_empty();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'is_on_sale': 
				if( $product !== false ) {
					$value = $product->is_on_sale();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'is_in_stock': 
				if( $product !== false ) {
					$value = $product->is_in_stock();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_low_stock': 
				if( $product !== false ) {
					$value = \BricksUltimate\WooCommerce::has_low_stock($product);
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'manage_stock': 
				if( $product !== false ) {
					$value = false;
					
					if ( ! $product->is_type('variable') ) {
						$value = $product->managing_stock();
					}

					if ( $product->is_type('variable') ) {
						foreach ( $product->get_visible_children() as $key => $variation_id ) {
							$variation = wc_get_product( $variation_id );
							if( $variation->managing_stock() ) {
								$value = $variation->managing_stock();
								break;
							}
						}
					}

					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'stock_quantity': 
				if( $product !== false && $product->managing_stock() ) {
					$value = $product->get_stock_quantity();
					$condition_met = self::execute_math_logic( $condition, $value );
				} else {
					$condition_met = true;
				}
				break;

			case 'backordered': 
				if( $product !== false ) {
					$value = $product->is_on_backorder();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'is_featured_product': 
				if( $product !== false ) {
					$value = $product->is_featured();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'is_virtual_product': 
				if( $product !== false ) {
					$value = $product->is_virtual();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'is_product_downloadable': 
				if( $product !== false ) {
					$value = $product->is_downloadable();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'is_product_purchasable': 
				if( $product !== false ) {
					$value = $product->is_purchasable();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_additional_info': 
				if( $product !== false ) {
					$value = ( $product && ( $product->has_attributes() || apply_filters( 'wc_product_enable_dimensions_display', $product->has_weight() || $product->has_dimensions() ) ) ) ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_reviews': 
				if( $product !== false ) {
					$value = ! empty( $product->get_review_count() );
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_related_products': 
				if( $product !== false ) {
					$value = sizeof( \wc_get_related_products( $product->get_id(), 2, $product->get_upsell_ids() ) )  > 0 ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_recently_viewed_products':
				$viewed_products = ! empty( $_COOKIE['woocommerce_recently_viewed'] ) ? (array) explode( '|', wp_unslash( $_COOKIE['woocommerce_recently_viewed'] ) ) : array(); // @codingStandardsIgnoreLine
				$viewed_products = array_reverse( array_filter( array_map( 'absint', $viewed_products ) ) );

				$condition_met = self::execute_true_false_logic( $condition, ! empty( $viewed_products ) );
				break;

			case 'has_upsell_products': 
				if( $product !== false ) {
					$value = sizeof( $product->get_upsell_ids() )  > 0 ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_cross_sell_products': 
				if( $product !== false ) {
					$value = sizeof( $product->get_cross_sell_ids() )  > 0 ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_linked_variations':
				global $wpdb;

				$post_type = \BricksUltimate\WooCommerce\WooLinkedProducts::get_instance()->post_type;

				// Prepare the SQL query
				$query = $wpdb->prepare(
					"
					SELECT SQL_CALC_FOUND_ROWS posts.ID 
					FROM {$wpdb->posts} AS posts
					INNER JOIN {$wpdb->postmeta} AS meta ON posts.ID = meta.post_id
					WHERE posts.post_type = %s
					AND posts.post_status = 'publish'
					AND meta.meta_key = '_bulv_products'
					AND meta.meta_value LIKE %s
					ORDER BY posts.post_date DESC
					LIMIT 1
					",
					$post_type,
					'%"' . (int) $post_id . '"%'
				);

				// Execute the query
				$result = $wpdb->get_row($query);
				if( $result ) {
					$products = get_post_meta( $result->ID, '_bulv_products', true );
					$value = ! empty( $products ) ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				} else {
					$condition_met = false;
				}
			break;

			case 'is_purchased': 
				if ( is_user_logged_in() && $product !== false ) {
					$customer = wp_get_current_user();
					$value = wc_customer_bought_product( $customer->user_email, $customer->ID, $product->get_id() );
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_weight': 
				if( $product !== false ) {
					$value = $product->has_weight();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_dimensions': 
				if( $product !== false ) {
					$value = $product->has_dimensions();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'product_sku': 
				if( $product !== false ) {
					$value = esc_html( $product->get_sku() );
					$condition_met = self::execute_equal_logic( $condition, $value );
				}
				break;

			case 'has_sale_start_date': 
				if( $product !== false ) {
					$value = is_object( $product->get_date_on_sale_from() ) ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_sale_end_date': 
				if( $product !== false ) {
					$value = is_object( $product->get_date_on_sale_to() ) ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case "has_shipping_methods":
				$value = false;

				if ( ! is_null( WC()->cart ) && WC()->cart->needs_shipping_address() && WC()->cart->show_shipping() ) {
					$value = true;
				}

				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'applied_coupon': 
				if( ! is_null( WC()->cart ) && isset( $condition['value'] ) ) {
					$value = in_array( strtolower( $condition['value'] ), WC()->cart->get_applied_coupons() ) ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'is_coupon_expired': 
				$coupon = false;

				if( is_object( $object ) && get_post_type( $object->ID ) == 'shop_coupon' ) {
					$coupon = new \WC_Coupon( $object->ID );
				} elseif( isset( $condition['value'] ) ) {
					$coupon = new \WC_Coupon( $condition['value'] );
				}

				if( is_object( $coupon ) && $coupon->get_date_expires() ) {
					$value = time() > $coupon->get_date_expires()->getTimestamp();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'is_coupon_valid': 
				$coupon = false;

				if( is_object( $object ) && get_post_type( $object->ID ) == 'shop_coupon' ) {
					$coupon = new \WC_Coupon( $object->ID );
				} elseif( isset( $condition['value'] ) ) {
					$coupon = new \WC_Coupon( $condition['value'] );
				}

				$value = is_object( $coupon ) && $coupon->is_valid();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'individual_use': 
				$coupon = false;

				if( is_object( $object ) && get_post_type( $object->ID ) == 'shop_coupon' ) {
					$coupon = new \WC_Coupon( $object->ID );
				} elseif( isset( $condition['value'] ) ) {
					$coupon = new \WC_Coupon( $condition['value'] );
				}

				$value = is_object( $coupon ) && $coupon->get_individual_use();
				$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'order_status': 
				$order = self::getCustomerOrder();

				if( $order ) {
					$value = 'wc-' . $order->get_status();
					$condition_met = self::execute_equal_logic( $condition, $value );
				}
				break;

			case 'payment_gateway': 
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = $order->get_payment_method();
					$condition_met = self::execute_equal_logic( $condition, $value );
				} else {
					$chosen_payment_method = \WC()->session->get( 'chosen_payment_method' );
					$condition_met = self::execute_equal_logic( $condition, $chosen_payment_method );
				}
				break;

			case 'has_previous_orders':
					global $wp;
					$current_page = isset( $wp->query_vars['orders'] ) && ! empty( $wp->query_vars['orders'] ) ? absint( $wp->query_vars['orders'] ) : 1;
					$value = ( 1 !== $current_page );
					$condition_met = self::execute_true_false_logic( $condition, $value );
				break;

			case 'has_next_orders':
				if( isset( $GLOBALS['totalOrdersPages'] ) ) {
					global $wp;
					$current_page = isset( $wp->query_vars['orders'] ) && ! empty( $wp->query_vars['orders'] ) ? absint( $wp->query_vars['orders'] ) : 1;
					$value = ( $current_page !== intval( $GLOBALS['totalOrdersPages'] ) );
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'needs_payment': 
				$order = self::getCustomerOrder();

				if( is_object( $order ) ) {
					$value = $order->needs_payment();
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'has_discount': 
				$order = self::getCustomerOrder();

				if( is_object( $order ) ) {
					$value = ( $order->get_total_discount() > 0 );
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'discount_amount': 
				$order = self::getCustomerOrder();

				if( is_object( $order ) ) {
					$value = (float) $order->get_total_discount();
					$condition_met = self::execute_math_logic( $condition, $value );
				}

				break;

			case 'has_shipping_cost': 
				$order = self::getCustomerOrder();

				if( is_object( $order ) ) {
					$value = ( $order->get_shipping_method() ) ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}

				break;

			case 'shipping_cost': 
				$order = self::getCustomerOrder();

				if( is_object( $order ) ) {
					$value = (float) $order->get_shipping_total();
					$condition_met = self::execute_math_logic( $condition, $value );
				}

				break;

			case 'has_customer_note': 
				$order = self::getCustomerOrder();

				if( is_object( $order ) ) {
					$value = ( $order->get_customer_note() ) ? true : false;
					$condition_met = self::execute_true_false_logic( $condition, $value );
				}
				break;

			case 'bought_specific_products': 
				$order = self::getCustomerOrder();

				if( is_object( $order ) ) {
					$ids = strstr( $condition['value'], ',' ) ? explode( ",", $condition['value'] ) : [ $condition['value'] ];
					$boughts = [];
					foreach ( $order->get_items() as $item ) {
						$product = $item->get_product();
						if ( $product && $product->get_parent_id() ) {
							$boughts[] = $product->get_parent_id();
						}

						if( $product && $product->get_id() ) {
							$boughts[] = $product->get_id();
						}
					}

					$condition_met = self::execute_true_false_logic( $condition, array_intersect($ids, $boughts) ? true : false );
				}
				break;
		}

		return $condition_met;
	}

	public function get_woo_conditions() {
		$tags = [];

		$tags['product_in_cart'] = [
			'label' => esc_html__( 'Product in cart', 'woocommerce' ),
			'group' => 'buwooprd',
			'type' 	=> 'truefalsetext',
			'placeholder' => esc_html__('24,12,45', 'bricksultimate'),
			'info' => esc_html__('Enter product ids with comma separator', 'bricksultimate'),
		];

		$tags['is_shop_page'] = [
			'label' => esc_html__( 'Is shop page?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_single_product_page'] = [
			'label' => esc_html__( 'Is single product page?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_cart_page'] = [
			'label' => esc_html__( 'Is cart page?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_checkout_page'] = [
			'label' => esc_html__( 'Is checkout page?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['order_pay_page'] = [
			'label' => esc_html__( 'Is order pay page?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_account_page'] = [
			'label' => esc_html__( 'Is my account page?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['thank_you_page'] = [
			'label' => esc_html__( 'Is thank you page?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_on_sale'] = [
			'label' => esc_html__( 'Is on sale?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_in_stock'] = [
			'label' => esc_html__( 'Is in stock?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['manage_stock'] = [
			'label' => esc_html__( 'Is stock managed?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_low_stock'] = [
			'label' => esc_html__( 'Is low stock?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['backordered'] = [
			'label' => esc_html__( 'Is in backorder?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_cart_empty'] = [
			'label' => esc_html__( 'Is cart empty?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_featured_product'] = [
			'label' => esc_html__( 'Is featured product?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_virtual_product'] = [
			'label' => esc_html__( 'Is virtual product?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_product_downloadable'] = [
			'label' => esc_html__( 'Is product downloadable?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_product_purchasable'] = [
			'label' => esc_html__( 'Is product purchasable?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['is_purchased'] = [
			'label' => esc_html__( 'Is already purchased?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_additional_info'] = [
			'label' => esc_html__( 'Has additional information?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_reviews'] = [
			'label' => esc_html__( 'Has reviews?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_related_products'] = [
			'label' => esc_html__( 'Has related products?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_recently_viewed_products'] = [
			'label' => esc_html__( 'Has recently viewed products?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_upsell_products'] = [
			'label' => esc_html__( 'Has upsell products?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_cross_sell_products'] = [
			'label' => esc_html__( 'Has cross-sell products?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_linked_variations'] = [
			'label' => esc_html__( 'Has linked variations?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_weight'] = [
			'label' => esc_html__( 'Has weight?', 'woocommerce' ),
			'group' => 'buwoo',
		];
		$tags['has_dimensions'] = [
			'label' => esc_html__( 'Has dimensions?', 'woocommerce' ),
			'group' => 'buwoo',
		];

		$tags['has_sale_start_date'] = [
			'label' => esc_html__( 'Has sale start date?', 'woocommerce' ),
			'group' => 'buwoo',
		];

		$tags['has_sale_end_date'] = [
			'label' => esc_html__( 'Has sale end date?', 'woocommerce' ),
			'group' => 'buwoo',
		];

		$tags['has_shipping_methods'] = [
			'label' => esc_html__( 'Has shipping methods?', 'woocommerce' ),
			'group' => 'buwoo',
		];

		//* Coupons
		$tags['applied_coupon'] = [
			'label' => esc_html__( 'Is coupon applied?', 'woocommerce' ),
			'group' => 'buwcpn',
			'type' 	=> 'truefalsetext',
			'info'	=> esc_html__( 'Enter coupon code. Leave empty if you are using the coupon loop.', 'bricksultimate' )
		];

		$tags['is_coupon_expired'] = [
			'label' => esc_html__( 'Is coupon expired?', 'woocommerce' ),
			'group' => 'buwcpn',
			'type' 	=> 'truefalsetext',
			'info'	=> esc_html__( 'Enter coupon code. Leave empty if you are using the coupon loop.', 'bricksultimate' )
		];

		$tags['is_coupon_valid'] = [
			'label' => esc_html__( 'Is coupon valid?', 'woocommerce' ),
			'group' => 'buwcpn',
			'type' 	=> 'truefalsetext',
			'info'	=> esc_html__( 'Enter coupon code. Leave empty if you are using the coupon loop.', 'bricksultimate' )
		];

		$tags['individual_use'] = [
			'label' => esc_html__( 'Individual use only', 'woocommerce' ),
			'group' => 'buwcpn',
			'type' 	=> 'truefalsetext',
			'info'	=> esc_html__( 'Enter coupon code. Leave empty if you are using the coupon loop.', 'bricksultimate' )
		];

		//* Orders
		$tags['order_status'] = [
			'label' => esc_html__( 'Order status', 'woocommerce' ),
			'group' => 'buwcord',
			'type' 	=> 'dropdown2nd',
			'options' => \wc_get_order_statuses()
		];

		$tags['has_previous_orders'] = [
			'label' => esc_html__( 'Has previous orders?', 'woocommerce' ),
			'group' => 'buwcord',
		];

		$tags['has_next_orders'] = [
			'label' => esc_html__( 'Has next orders?', 'woocommerce' ),
			'group' => 'buwcord',
			'type' 	=> 'truefalsetext',
			'placeholder' => esc_html__('ipqasj', 'bricksultimate'),
			'info' => esc_html__('Enter 6 digit customer orders loop builder element id', 'bricksultimate'),
		];

		$tags['needs_payment'] = [
			'label' => esc_html__( 'Needs payment?', 'woocommerce' ),
			'group' => 'buwcord',
			'type' 	=> 'truefalse',
			'info' => esc_html__('It needs for Pay order button', 'bricksultimate'),
		];

		$tags['has_discount'] = [
			'label' => esc_html__( 'Has discount?', 'woocommerce' ),
			'group' => 'buwcord',
		];

		$tags['discount_amount'] = [
			'label' => esc_html__( 'Discount amount', 'woocommerce' ),
			'group' => 'buwcord',
			'type' 	=> 'math',
			'info'  => esc_html__('It is returning the float value. You will enter the float value into the input box.', 'bricksultimate')
		];

		$tags['has_shipping_cost'] = [
			'label' => esc_html__( 'Has shipping cost?', 'woocommerce' ),
			'group' => 'buwcord',
		];

		$tags['shipping_cost'] = [
			'label' => esc_html__( 'Shipping cost', 'woocommerce' ),
			'group' => 'buwcord',
			'type' 	=> 'math',
			'info'  => esc_html__('It is returning the float value. You will enter the float value into the input box.', 'bricksultimate')
		];

		$tags['has_customer_note'] = [
			'label' => esc_html__( 'Has customer note?', 'woocommerce' ),
			'group' => 'buwcord',
		];

		$tags['bought_specific_products'] = [
			'label' => esc_html__( 'Bought specific products', 'bricksultimate' ),
			'group' => 'buwcord',
			'type' 	=> 'truefalsetext',
			'placeholder' => esc_html__('24,12,45', 'bricksultimate'),
			'info' => esc_html__('Enter product ids with comma separator', 'bricksultimate'),
		];

		$tags['payment_gateway'] = [
			'label' => esc_html__( 'Payment gateway', 'woocommerce' ),
			'group' => 'buwcord',
			'type' 	=> 'dropdown2nd',
			'options' => $this->get_payment_method_options()
		];

		return $tags;
	}

	public function get_payment_method_options() {
		$payment_options = ['none' => esc_html__('Select', 'bricks')];
		$payment_gateways = \WC()->payment_gateways->get_available_payment_gateways();
		if( ! empty( $payment_gateways ) ) {
			foreach ( $payment_gateways as $id => $gateway ) {
				$payment_options[ $id ] = esc_attr( $gateway->get_title() );
			}
		}

		return $payment_options;
	}

	public static function getCustomerOrder() {
		return \BricksUltimate\DynamicData\ProviderUltimate::getCustomerOrder();
	}
}