<?php
namespace BricksUltimate\Elements;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use BricksUltimate\Helpers;

class Lightbox extends \Bricks\Element {
	public $category 		= 'ultimate';
	public $name 			= 'bu-lightbox';
	public $icon 			= 'ti-layout-media-left-alt';
	public $tag 			= 'div';
	public $nestable 		= true;

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Simple Lightbox', 'bricksultimate' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['popup']    = array(
			'title' => esc_html__( 'Popup box', 'bricksultimate' ),
			'tab'   => 'content',
		);

		$this->control_groups['popupBD']    = array(
			'title' => esc_html__( 'Backdrop', 'bricksultimate' ),
			'tab'   => 'content',
		);
	}

	// Set builder controls
	public function set_controls() {
		$this->controls['builder_mode'] = [
			'tab' 	=> 'content',
			'type' 	=> 'checkbox',
			'label' => __('Hide In-Builder Editor?', 'bricksultimate'),
			'inline' => true,
		];

		$this->controls['previewOpenSep'] = [
			'type'  => 'separator',
			'label' => esc_html__( 'Dynamic data preview on the builder editor', 'bricks' ),
		];

		$this->controls['previewId'] = [
			'tab' 	=> 'content',
			'type' 	=> 'text',
			'label' => __('Query loop builder element ID', 'bricksultimate'),
		];

		$this->controls['previewCloseSep'] = [
			'type'  => 'separator',
			'label' => '',
		];

		$this->controls['disable_scroll'] = [
			'tab' 	=> 'content',
			'type' 	=> 'checkbox',
			'label' => __('Disable scroll?', 'bricksultimate'),
			'inline' => true,
		];

		$this->controls['tag'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'HTML tag', 'bricks' ),
			'type'        => 'select',
			'options'     => [
				'div'     => 'div',
				'section' => 'section',
				'article' => 'article',
				'nav'     => 'nav',
				'aside'   => 'aside',
			],
			'lowercase'   => true,
			'inline'      => true,
			'placeholder' => $this->tag ? $this->tag : 'div',
			'fullAccess'  => true,
		];

		$this->controls['customTag'] = [
			'tab'            => 'content',
			'label'          => esc_html__( 'Custom tag', 'bricks' ),
			'type'           => 'text',
			'inline'         => true,
			'hasDynamicData' => false,
			'placeholder'    => 'div',
			'required'       => [ 'tag', '=', 'custom' ],
		];

		$this->controls['trigger_type'] = [
			'tab' 			=> 'content',
			'label' 		=> esc_html__( 'Trigger on', 'bricksultimate' ),
			'type' 			=> 'select',
			'options' 		=> [
				'click'     => esc_html__('Click', 'bricksultimate'),
				'hover' 	=> esc_html__('Hover', 'bricksultimate')
			],
			'inline' 		=> true,
			'placeholder' 	=> 'click',
		];

		$this->controls['trigger_selector'] = [
			'tab' 	=> 'content',
			'type' 	=> 'text',
			'label' => __('Trigger selectors', 'bricksultimate'),
		];

		//* popup box
		$this->controls['popupContentWidth'] = [
			'group' => 'popup',
			'label' => esc_html__( 'Width', 'bricks' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'width',
					'selector' => '.bu-lb-content',
				],
			],
		];

		$this->controls['popupContentHeight'] = [
			'group' => 'popup',
			'label' => esc_html__( 'Height', 'bricks' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'height',
					'selector' => '.bu-lb-content',
				],
			],
		];

		$this->controls['popupBackground'] = [
			'group' => 'popup',
			'label' => esc_html__( 'Background', 'bricks' ),
			'type'  => 'color',
			'css'   => [
				[
					'property' => 'background',
					'selector' => '.bu-lb-content',
				],
			],
		];

		$this->controls['popupContentBorder'] = [
			'group' => 'popup',
			'label' => esc_html__( 'Border', 'bricks' ),
			'type'  => 'border',
			'css'   => [
				[
					'property' => 'border',
					'selector' => '.bu-lb-content',
				],
			],
		];

		$this->controls['popupContentBoxShadow'] = [
			'group' => 'popup',
			'label' => esc_html__( 'Box shadow', 'bricks' ),
			'type'  => 'box-shadow',
			'css'   => [
				[
					'property' => 'box-shadow',
					'selector' => '.bu-lb-content',
				],
			],
		];

		$this->controls['popupSep1'] = [
			'group' => 'popup',
			'label' => '',
			'type'  => 'separator',
		];

		$this->controls['popupJustifyConent'] = [
			'group'        => 'popup',
			'label'        => esc_html__( 'Align main axis', 'bricks' ),
			'tooltip'      => [
				'content'  => 'justify-content',
				'position' => 'top-left',
			],
			'type'         => 'justify-content',
			'inline'       => true,
			'exclude'      => [
				'space',
			],
			'css'          => [
				[
					'property' => 'justify-content',
				],
			],
		];

		$this->controls['popupAlignItems'] = [
			'group'   => 'popup',
			'label'   => esc_html__( 'Align cross axis', 'bricks' ),
			'tooltip' => [
				'content'  => 'align-items',
				'position' => 'top-left',
			],
			'type'    => 'align-items',
			'inline'  => true,
			'exclude' => [
				'stretch',
			],
			'css'     => [
				[
					'property' => 'align-items',
				],
			],
		];

		$this->controls['popupSep2'] = [
			'group' => 'popup',
			'label' => '',
			'type'  => 'separator',
		];

		$this->controls['popupMargin'] = [
			'group' => 'popup',
			'label' => esc_html__( 'Margin', 'bricks' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'margin',
					'selector' => '.bu-lb-content',
				],
			],
		];

		$this->controls['popupContentPadding'] = [
			'group' => 'popup',
			'label' => esc_html__( 'Padding', 'bricks' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.bu-lb-content',
				],
			],
			'placeholder' => [
				'top'    => '30px',
				'right'  => '30px',
				'bottom' => '30px',
				'left'   => '30px',

			],
		];

		//* Backdrop
		$this->controls['disableBD'] = [
			'group' => 'popupBD',
			'type' 	=> 'checkbox',
			'label' => __('Disable backdrop?', 'bricksultimate'),
			'inline' => true,
		];

		$this->controls['popupBDBackground'] = [
			'group' => 'popupBD',
			'label' => esc_html__( 'Background', 'bricks' ),
			'type'  => 'color',
			'css'   => [
				[
					'property' => 'background',
					'selector' => '&.bu-lb-has-backdrop',
				],
			],
		];

		$this->controls['bdCloseLightbox'] = [
			'group' => 'popupBD',
			'label' => esc_html__( 'Close lightbox when clicking on the backdrop?', 'bricks' ),
			'type'  => 'select',
			'options' => [
				'yes' 	=> __('Yes'),
				'no' 	=> __('No')
			],
			'default' => 'yes'
		];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_style( 'bu-slb', Helpers::get_asset_url('css') . 'slb.min.css', [], filemtime(Helpers::get_asset_path('css') . 'slb.min.css') );
		/*wp_enqueue_script( 'bu-ocp', Helpers::get_asset_url('js') . 'ocp.min.js', [], filemtime(Helpers::get_asset_path('js') . 'ocp.min.js'), true );*/
	}

	// Render element HTML
	public function render() {
		$element  = $this->element;
		$settings = ! empty( $element['settings'] ) ? $element['settings'] : [];
		$config = [];

		if( empty( $settings['disableBD'] ) ) {
			$this->set_attribute( '_root', 'class', 'bu-lb-has-backdrop' );
		}
		$config['bdCloseLightbox'] = Helpers::get_value( $settings, 'bdCloseLightbox', 'yes' );

		$this->set_attribute( '_root', 'class', 'bu-simple-lightbox' );
		$this->set_attribute( '_root', 'class', 'bu-lightbox--close' );
		echo "<{$this->tag} {$this->render_attributes( '_root' )}>";

		$this->set_attribute( 'bu_lb_content', 'class', 'bu-lb-content' );
		$output = "<div {$this->render_attributes( 'bu_lb_content' )}>";

		$children = $element['children'];
		$post_id = $this->post_id;

		if( $children && Helpers::isBricksBuilderActive() ) {
			if( isset( $settings['previewId'] ) ) {
				$query = new \Bricks\Query( [ 'id' => $settings['previewId'] ] );
				if( $query->query_result->have_posts() ) {
					$post_id = $query->query_result->posts[0]->ID;
				}
				$query->destroy();
				unset( $query );
			}

			foreach ( $children as $child_id ) {
				$childSettings = \Bricks\Helpers::get_element_data( $this->post_id, $child_id );
				if ( $childSettings ) {
					$output .= $this->builder_render_child_element( $childSettings['element'], $post_id );
				}
			}
		} else {
			
			$output .= \Bricks\Frontend::render_children( $this );
			
		}
		$output .= "</div>";
		echo $output;

		echo "</{$this->tag}>";
	}

	public function builder_render_child_element( $element, $post_id ) {
		// Check: Get global element settings
		$global_settings = \Bricks\Helpers::get_global_element( $element, 'settings' );

		if ( is_array( $global_settings ) ) {
			$element['settings'] = $global_settings;
		}

		$element_class_name = isset( \Bricks\Elements::$elements[ $element['name'] ]['class'] ) ? \Bricks\Elements::$elements[ $element['name'] ]['class'] : $element['name'];

		if ( class_exists( $element_class_name ) ) {
			$element['themeStyleSettings'] = \Bricks\Theme_Styles::$active_settings;

			$element_instance = new $element_class_name( $element );
			$element_instance->load();

			// Enqueue element styles/scripts & render element
			ob_start();
			$this->init_element( $post_id, $element_instance );
			return ob_get_clean();
		}
	}

	public function init_element( $post_id, $element ) {
		// Enqueue scripts & styles
		$element->enqueue_scripts();

		$element->set_post_id( $post_id );

		// Set root attributes
		$element->set_root_attributes();

		$render_element = true;

		if ( ! empty( $element->settings['_conditions'] ) ) {
			$render_element = \Bricks\Conditions::check( $element->settings['_conditions'], $element );
		}

		$render_element = apply_filters( 'bricks/element/render', $render_element, $element );

		if ( $render_element ) {
			$element->settings = apply_filters( 'bricks/element/settings', $element->settings, $element );

			$element->post_id = $post_id;

			$element->render();
		}
	}

	public static function render_builder() {
		?>
		<script type="text/x-template" id="tmpl-bricks-element-bu-lightbox">
			<component 
				:is="'div'" 
				v-if="!settings.builder_mode" 
				class="bu-simple-lightbox" 
				:class="[{'bu-lb-has-backdrop' : !settings.disableBD}]" 
				:data-lb-config="JSON.stringify({
					bdCloseLightbox: settings.bdCloseLightbox
				})"
			>
				<div class="bu-lb-content">
					<bricks-element-children :element="element" />
				</div>
			</component>
		</script>
	<?php 
	}
}