<?php
namespace BricksUltimate\Elements;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use BricksUltimate\Helpers;

class OffCanvas extends \Bricks\Element {
	public $category 		= 'ultimate';
	public $name 			= 'bu-off-canvas';
	public $icon 			= 'ti-layout-sidebar-left';
	public $tag 			= 'section';
	public $nestable 		= true;
	public $output 			= '';
	public $scripts 		= [ 'buOffCanvas' ];

	// Return localized element label
	public function get_label() {
		return esc_html__( '(BU)Off-Canvas', 'bricksultimate' );
	}

	public function get_keywords() {
		return [ 'canvas', 'off canvas', 'panel' ];
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['panel']    = array(
			'title' => esc_html__( 'Panel', 'bricksultimate' ),
			'tab'   => 'content',
		);

		$this->control_groups['backdrop']    = array(
			'title' => esc_html__( 'Backdrop', 'bricksultimate' ),
			'tab'   => 'content',
		);

		$this->control_groups['scrollbar']    = array(
			'title' => esc_html__( 'Scrollbar', 'bricksultimate' ),
			'tab'   => 'content',
		);

		$this->control_groups['minicart']    = array(
			'title' => esc_html__( 'Mini Cart', 'bricksultimate' ),
			'tab'   => 'content',
		);

		$this->control_groups['miniwsl']    = array(
			'title' => esc_html__( 'Mini Wishlist', 'bricksultimate' ),
			'tab'   => 'content',
		);

		$this->control_groups['minicomp']    = array(
			'title' => esc_html__( 'Mini Compare', 'bricksultimate' ),
			'tab'   => 'content',
		);
	}

	// Set builder controls
	public function set_controls() {
		$this->controls['builder_mode'] = [
			'tab' 	=> 'content',
			'type' 	=> 'checkbox',
			'label' => __('Hide In-Builder Editor?', 'bricksultimate'),
			'inline' => true
		];

		$this->controls['trigger_on'] = [
			'tab' 	=> 'content',
			'type' 	=> 'select',
			'label' => __('Trigger on', 'bricksultimate'),
			'options' => [
				'click' 	=> esc_html__('Click'),
				'mouseover' => esc_html__('Hover'),
				'scroll' 	=> esc_html__('Scroll')
			],
			'inline' 	=> true,
			'medium' 	=> true,
			'default' 	=> 'click'
		];

		$this->controls['offset'] = [
			'tab' 		=> 'content',
			'type'      => 'number',
			'label'     => esc_html__('Offset(px)', 'bricksultimate'),
			'min'		=> 0,
			'max'  		=> 10000,
			'step' 		=> 20,
			'placeholder' => 0,
			'unitless' 	=> true,
			'small' 	=> true,
			'inline' 	=> true,
			'required' 	=> ['trigger_on', '=', "scroll"]
		];

		$this->controls['trigger_selector'] = [
			'tab' 	=> 'content',
			'type' 	=> 'text',
			'label' => __('Trigger selector', 'bricksultimate'),
			'required' 	=> ['trigger_on', '!=', "scroll"]
		];

		$this->controls['slide_speed'] = [
			'tab' 		=> 'content',
			'type'      => 'number',
			'label'     => esc_html__('Transition duration(sec)', 'bricksultimate'),
			'unit' 		=> 's',
			'min'		=> 0,
			'max'  		=> 10,
			'step' 		=> .1,
			'default'   => 0.5,
			'small' 	=> true,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> '.bu-oc-panel',
					'property' 	=> 'transition-duration'
				]
			]
		];

		$this->controls['disable_scroll'] = [
			'tab' 	=> 'content',
			'type' 	=> 'checkbox',
			'label' => __('Disable scroll', 'bricksultimate'),
			'inline' => true,
		];

		$this->controls['hook'] = [
			'tab' 		=> 'content',
			'type' 		=> 'select',
			'label' 	=> __('Placement', 'bricksultimate'),
			'options' 	=> [
				'none' => esc_html__('Current position', 'bricksultimate'),
				'render_header' => esc_html__('After <body> Tag', 'bricksultimate'),
				'bricks_after_site_wrapper' => esc_html__('Above </body> Tag', 'bricksultimate'),
			],
			'default' 	=> 'none',
			'inline' 	=> true,
			'medium' 	=> true
		];


		/**
		 * Group - Panel settings
		 */
		$this->controls['ocp_position'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Position', 'bricks'),
			'options' 	=> [
				'bu-ocp-left' => esc_html__('Left'),
				'bu-ocp-right' => esc_html__('Right'),
				'bu-ocp-top' => esc_html__('Top'),
				'bu-ocp-bottom' => esc_html__('Bottom'),
			],
			'default' 	=> 'bu-ocp-right',
			'inline' 	=> true,
			'medium' 	=> true
		];

		$selector = '.bu-oc-panel';

		$this->controls['width'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => 'number',
			'label'     => esc_html__('Width', 'bricks'),
			'units' 	=> true,
			'min' 		=> 0,
			'step' 		=> 1,
			'inline' 	=> true,
			'placeholder' => '300px',
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'width'
				]
			]
		];

		$this->controls['height'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => 'number',
			'label'     => esc_html__('Height', 'bricks'),
			'units' 	=> true,
			'min' 		=> 0,
			'step' 		=> 1,
			'inline' 	=> true,
			'placeholder' => '300px',
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'height'
				]
			]
		];

		$this->controls['padding'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => version_compare( BRICKS_VERSION, '1.5', '>') ? 'spacing' : 'dimensions',
			'label'     => esc_html__('Spacing', 'bricks'),
			'default' 	=> [
				'top' 		=> 18,
				'right' 	=> 18,
				'bottom' 	=> 18,
				'left' 		=> 18,
			],
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'padding'
				]
			]
		];

		$this->controls['panelBg'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => 'color',
			'label'     => esc_html__('Background', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'background-color'
				]
			]
		];

		$this->controls['panelBg'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => 'color',
			'label'     => esc_html__('Background', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'background-color'
				]
			]
		];

		$this->controls['panelBrd'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => 'border',
			'label'     => esc_html__('Border', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'border'
				]
			]
		];

		$this->controls['panelBS'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => 'box-shadow',
			'label'     => esc_html__('Box shadow', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'box-shadow'
				]
			]
		];

		$this->controls['panelSep'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => 'separator',
			'label'     => esc_html__('Extra Actions', 'bricks'),
		];

		$this->controls['escBtn'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type' 		=> 'checkbox',
			'label' 	=> __('Prevent Esc to close panel', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true
		];

		$this->controls['closeHashLink'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type' 		=> 'checkbox',
			'label' 	=> __('Close after clicking hash link', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true
		];

		$this->controls['push_content'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type' 		=> 'checkbox',
			'label' 	=> __('Push Body Content', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true,
			'required' 	=> ['trigger_on', '!=', "scroll"]
		];

		$this->controls['reveal_panel'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type' 		=> 'checkbox',
			'label' 	=> __('Reveal Panel on Page Load', 'bricksultimate'),
			'info' 		=> __('Panel will automatically slide out on page load.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true
		];
		
		$this->controls['delay'] = [
			'tab' 		=> 'content',
			'group' 	=> 'panel',
			'type'      => 'number',
			'label'     => esc_html__('Delay in ms', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 50,
			'default'   => 1200,
			'medium' 	=> true,
			'inline' 	=> true,
			'required' 	=> ['reveal_panel', '=', true]
		];


		/**
		 * Reveal mini cart
		 */
		$this->controls['revealMiniCart'] = [
			'group' 	=> 'minicart',
			'type' 		=> 'checkbox',
			'label' 	=> __('Reveal', 'bricksultimate'),
			'description' => __('The panel will automatically open when a product is added to the cart.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true
		];

		$this->controls['stopAutoClose'] = [
			'group' 	=> 'minicart',
			'type' 		=> 'checkbox',
			'label' 	=> __('Disable auto close', 'bricksultimate'),
			'info' 		=> __('The panel will not close automatically on all devices.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true,
			'required' 	=> ['revealMiniCart', '=', true]
		];

		$this->controls['stopRevealMobile'] = [
			'group' 	=> 'minicart',
			'type' 		=> 'checkbox',
			'label' 	=> __('Disable autoclose(Mobile only)', 'bricksultimate'),
			'info' 		=> __('Panel will not close automatically on mobile.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true,
			'default' 	=> true,
			'required' 	=> ['revealMiniCart', '=', true]
		];

		$this->controls['mcdelay'] = [
			'group' 	=> 'minicart',
			'type'      => 'number',
			'label'     => esc_html__('Delay in ms', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 50,
			'default'   => 1200,
			'medium' 	=> true,
			'inline' 	=> true,
			'required' 	=> ['revealMiniCart', '=', true]
		];

		$this->controls['mcduration'] = [
			'group' 	=> 'minicart',
			'type'      => 'number',
			'label'     => esc_html__('Duration in ms', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 100,
			'default'   => 4500,
			'medium' 	=> true,
			'inline' 	=> true,
			'required' 	=> ['revealMiniCart', '=', true]
		];

		/**
		 * Group - Backdrop settings
		 */
		$selector = '.bu-ocp-backdrop';
		$this->controls['disableBackdrop'] = [
			'tab' 	=> 'content',
			'group' => 'backdrop',
			'type' 	=> 'checkbox',
			'label' => __('Disable backdrop', 'bricksultimate'),
			'inline' => true,
			'css' 	=> [
				[
					'property' 	=> 'display',
					'selector' 	=> $selector,
					'value' 	=> 'none'
				]
			]
		];

		$this->controls['disableBackdropClick'] = [
			'tab' 	=> 'content',
			'group' => 'backdrop',
			'type' 	=> 'checkbox',
			'label' => __('Prevent click to close panel', 'bricksultimate'),
			'info' 	=> __('Panel will not close after clicking on the backdrop', 'bricksultimate'),
			'reset' => true,
		];

		$this->controls['backdropBG'] = [
			'tab' 		=> 'content',
			'group' 	=> 'backdrop',
			'type'      => 'color',
			'label'     => esc_html__('Background', 'bricks'),
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'background-color'
				]
			]
		];

		$this->controls['backdropTd'] = [
			'tab' 		=> 'content',
			'group' 	=> 'backdrop',
			'type'      => 'number',
			'label'     => esc_html__('Fade duration(sec)', 'bricksultimate'),
			'unit' 		=> 's',
			'min'		=> 0,
			'max'  		=> 10,
			'step' 		=> .1,
			'default'   => 0.5,
			'small' 	=> true,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'transition-duration'
				]
			]
		];

		/**
		 * Group - Scrollbar settings
		 */
		$this->controls['will_customize'] = [
			'tab' 		=> 'content',
			'group' 	=> 'scrollbar',
			'type' 		=> 'checkbox',
			'label' 	=> __('Customize scrollbar', 'bricksultimate'),
			'inline' 	=> true,
		];

		$this->controls['sbWidth'] = [
			'tab' 		=> 'content',
			'group' 	=> 'scrollbar',
			'type'      => 'number',
			'label'     => esc_html__('Width', 'bricks'),
			'units' 	=> true,
			'min' 		=> 0,
			'max' 		=> 20,
			'step' 		=> 1,
			'inline' 	=> true,
			'placeholder' => '6px',
			'css' 		=> [
				[
					'selector' 	=> '.bu-oc-panel::-webkit-scrollbar',
					'property' 	=> 'width'
				]
			],
			'required' => ['will_customize', '=', true]
		];

		$this->controls['sbColor'] = [
			'tab' 		=> 'content',
			'group' 	=> 'scrollbar',
			'type'      => 'color',
			'label'     => esc_html__('Bar color', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> '',
					'property' 	=> '--sb-light-color'
				]
			],
			'required' => ['will_customize', '=', true]
		];

		$this->controls['sbHandlerColor'] = [
			'tab' 		=> 'content',
			'group' 	=> 'scrollbar',
			'type'      => 'color',
			'label'     => esc_html__('Handler color', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> '',
					'property' 	=> '--sb-dark-color'
				]
			],
			'required' => ['will_customize', '=', true]
		];


		/**
		 * Reveal mini wishlist
		 */
		$this->controls['revealMiniWsl'] = [
			'group' 	=> 'miniwsl',
			'type' 		=> 'checkbox',
			'label' 	=> __('Reveal', 'bricksultimate'),
			'description' => __('The panel will automatically open when a product is added to the wishlist.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true
		];

		$this->controls['wslStopAutoClose'] = [
			'group' 	=> 'miniwsl',
			'type' 		=> 'checkbox',
			'label' 	=> __('Disable auto close', 'bricksultimate'),
			'info' 		=> __('The panel will not close automatically on all devices.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true,
			'required' 	=> ['revealMiniWsl', '=', true]
		];

		$this->controls['wslStopRevealMobile'] = [
			'group' 	=> 'miniwsl',
			'type' 		=> 'checkbox',
			'label' 	=> __('Disable autoclose(Mobile only)', 'bricksultimate'),
			'info' 		=> __('Panel will not close automatically on mobile.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true,
			'default' 	=> true,
			'required' 	=> ['revealMiniWsl', '=', true]
		];

		$this->controls['wslmcdelay'] = [
			'group' 	=> 'miniwsl',
			'type'      => 'number',
			'label'     => esc_html__('Delay in ms', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 50,
			'default'   => 1200,
			'medium' 	=> true,
			'inline' 	=> true,
			'required' 	=> ['revealMiniWsl', '=', true]
		];

		$this->controls['wslmcduration'] = [
			'group' 	=> 'miniwsl',
			'type'      => 'number',
			'label'     => esc_html__('Duration in ms', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 100,
			'default'   => 4500,
			'medium' 	=> true,
			'inline' 	=> true,
			'required' 	=> ['revealMiniWsl', '=', true]
		];


		/**
		 * Reveal mini compare
		 */
		$this->controls['revealMiniComp'] = [
			'group' 	=> 'minicomp',
			'type' 		=> 'checkbox',
			'label' 	=> __('Reveal', 'bricksultimate'),
			'description' => __('The panel will automatically open when a product is added to the compare list.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true
		];

		$this->controls['compStopAutoClose'] = [
			'group' 	=> 'minicomp',
			'type' 		=> 'checkbox',
			'label' 	=> __('Disable auto close', 'bricksultimate'),
			'info' 		=> __('The panel will not close automatically on all devices.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true,
			'required' 	=> ['revealMiniComp', '=', true]
		];

		$this->controls['compStopRevealMobile'] = [
			'group' 	=> 'minicomp',
			'type' 		=> 'checkbox',
			'label' 	=> __('Disable autoclose(Mobile only)', 'bricksultimate'),
			'info' 		=> __('Panel will not close automatically on mobile.', 'bricksultimate'),
			'inline' 	=> true,
			'reset' 	=> true,
			'default' 	=> true,
			'required' 	=> ['revealMiniComp', '=', true]
		];

		$this->controls['compmcdelay'] = [
			'group' 	=> 'minicomp',
			'type'      => 'number',
			'label'     => esc_html__('Delay in ms', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 50,
			'default'   => 1200,
			'medium' 	=> true,
			'inline' 	=> true,
			'required' 	=> ['revealMiniComp', '=', true]
		];

		$this->controls['compmcduration'] = [
			'group' 	=> 'minicomp',
			'type'      => 'number',
			'label'     => esc_html__('Duration in ms', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 100,
			'default'   => 4500,
			'medium' 	=> true,
			'inline' 	=> true,
			'required' 	=> ['revealMiniComp', '=', true]
		];

	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		/*if( bricks_is_builder() || bricks_is_builder_iframe() || bricks_is_builder_call() ){
			wp_enqueue_style( 'bu-ocp', Helpers::get_asset_url('css') . 'ocp.min.css', [], filemtime(Helpers::get_asset_path('css') . 'ocp.min.css') );
		}*/
		wp_enqueue_script( 'bu-ocp', Helpers::get_asset_url('js') . 'ocp.min.js', [], filemtime(Helpers::get_asset_path('js') . 'ocp.min.js'), true );
	}

	// Render element HTML
	public function render() {
		if( Helpers::isBricksBuilderEditor() )
			return;

		$hook = Helpers::get_value( $this->settings, 'hook', 'none' );
		
		$settings = $this->settings;
		$attr = $class = [];

		$class[] = Helpers::get_value( $settings, 'ocp_position', 'bu-ocp-right' );
		$class[] = 'bu-hide-panel screen-reader-text';
		$class[] = isset( $settings['will_customize'] ) ? 'bu-customize-sb' : '';
		
		$attr['panelWidth'] 	= Helpers::get_value( $settings, 'width', 300 );
		$attr['panelHeight'] 	= Helpers::get_value( $settings, 'height', 600 );
		$attr['disableScroll'] 	= isset( $settings['disable_scroll'] ) ? 'yes' : 'no';
		$attr['disableBackdropClick'] = isset( $settings['disableBackdropClick'] ) ? 'yes' : 'no';
		$attr['escBtn'] 		= isset( $settings['escBtn'] ) ? 'yes' : 'no';
		$attr['closeHashLink'] 	= isset( $settings['closeHashLink'] ) ? 'yes' : 'no';
		$attr['revealPanel'] 	= ! empty( $settings['reveal_panel'] ) ? 'yes' : 'no';

		//* mini-cart
		$attr['stopAutoClose'] = ! empty( $settings['stopAutoClose'] ) ? 'yes' : 'no';
		$attr['stopRevealMobile'] = ! empty( $settings['stopRevealMobile'] ) ? 'yes' : 'no';
		$attr['revealDelay'] 	= Helpers::get_value( $settings, 'delay', 1200 );
		$attr['revealMiniCart'] = ! empty( $settings['revealMiniCart'] ) ? 'yes' : 'no';
		$attr['revealMCDelay'] 	= Helpers::get_value( $settings, 'mcdelay', 1200 );
		$attr['revealMCDuration'] = Helpers::get_value( $settings, 'mcduration', 4500 );

		if( ! empty( $settings['revealMiniWsl'] ) ) {
			$attr['revealMiniWsl'] = 'yes';
			$attr['wslStopAutoClose'] = ! empty( $settings['wslStopAutoClose'] ) ? 'yes' : 'no';
			$attr['wslStopRevealMobile'] = ! empty( $settings['wslStopRevealMobile'] ) ? 'yes' : 'no';
			$attr['wslMCDelay'] 	= Helpers::get_value( $settings, 'wslmcdelay', 1200 );
			$attr['wslMCDuration'] = Helpers::get_value( $settings, 'wslmcduration', 4500 );
		}

		if( ! empty( $settings['revealMiniComp'] ) ) {
			$attr['revealMiniComp'] = 'yes';
			$attr['compStopAutoClose'] = ! empty( $settings['compStopAutoClose'] ) ? 'yes' : 'no';
			$attr['compStopRevealMobile'] = ! empty( $settings['compStopRevealMobile'] ) ? 'yes' : 'no';
			$attr['compMCDelay'] 	= Helpers::get_value( $settings, 'compmcdelay', 1200 );
			$attr['compMCDuration'] = Helpers::get_value( $settings, 'compmcduration', 4500 );
		}

		$attr['triggerSelector'] = Helpers::get_value( $settings, 'trigger_selector', false );
		$attr['triggerEvent'] 	= Helpers::get_value( $settings, 'trigger_on', 'click' );
		$attr['panelTd'] 		= Helpers::get_value( $settings, 'slide_speed', 0.5 ) * 1000;
		$attr['offset'] 		= Helpers::get_value( $settings, 'offset', 0 );
		$attr['ocpPosition'] 	= str_replace( 'bu-ocp-', '', Helpers::get_value( $settings, 'ocp_position', 'bu-ocp-right' ) );

		if( ! empty( $settings['push_content']) ) {
			$class[] = 'bu-push-content';
		}

		$this->set_attribute( '_root', 'class', $class );
		$this->set_attribute( '_root', 'data-ocp-config', wp_json_encode( $attr ) );

		$output = "<{$this->tag} {$this->render_attributes( '_root' )}>";

		if( ! isset( $settings['disableBackdrop'] ) )
			$output .= '<div class="bu-ocp-backdrop"></div>';

		$this->set_attribute( 'bu-oc-panel', 'class', "bu-oc-panel" );

		$output .= "<aside {$this->render_attributes( 'bu-oc-panel' )}>";
		
		// Render children elements (= individual items)
		$output .= \Bricks\Frontend::render_children( $this );

		$output .= '</aside>';

		$output .= "</{$this->tag}>";

		if( $hook == 'none' ) {
			echo $output;
		} else {
			$this->output = $output;
		}
	}

	public function offCanvasOutput() {
		echo $this->output;
	}

	public function add_actions() {
		$hook = Helpers::get_value( $this->settings, 'hook' );
		if( $hook != "none" ) {
			add_action( $hook, [$this, 'offCanvasOutput'], 15 );
		}
	}

	public static function render_builder() {?>
		<script type="text/x-template" id="tmpl-bricks-element-bu-off-canvas">
			<component 
				:is="'section'" 
				class="bu-off-canvas"
				:class="[{'bu-customize-sb':settings.will_customize,'bu-hide-panel':settings.builder_mode},settings.ocp_position || 'bu-ocp-right']" 
				:data-ocp-config="JSON.stringify({
					panelWidth: settings.hasOwnProperty('width') ? parseInt(settings.width) : 300,
					panelHeight: settings.hasOwnProperty('height') ? parseInt(settings.height) : 300,
					disableScroll: settings.hasOwnProperty('disable_scroll') ? 'yes' : 'no',
					revealPanel: settings.hasOwnProperty('reveal_panel') ? 'yes' : 'no',
					revealDelay: settings.hasOwnProperty('delay') ? parseInt(settings.delay) : 1200,
					triggerSelector: settings.hasOwnProperty('trigger_selector') ? settings.trigger_selector : false,
					triggerEvent: settings.hasOwnProperty('trigger_on') ? settings.trigger_on : 'click',
					panelTd: settings.hasOwnProperty('slide_speed') ? parseFloat(settings.slide_speed) * 1000 : 500,
					offset: settings.hasOwnProperty('offset') ? parseInt(settings.offset) : 0,
					ocpPosition: settings.ocp_position.replace('bu-ocp-', '')
				})"
			>
				<div class="bu-ocp-backdrop" v-if="!settings.disableBackdrop"></div>
				<aside class="bu-oc-panel">
					<bricks-element-children :element="element" />
				</aside>
			</component>
		</script>
		<?php
	}
}