<?php
namespace BricksUltimate\Elements;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use BricksUltimate\{Helpers,WooCommerce};

class WooFreeShippingNotice extends \Bricks\Element {
	public $category 		= 'free shipping notice';
	public $name 			= 'bu-free-shipping-notice';
	public $icon 			= 'ti-align-justify';
	public $scripts 		= ['bufsnText'];

	// Return localized element label
	public function get_label() {
		return esc_html__( '(BU)Free Shipping Text', 'bricksultimate' );
	}

	// Set builder controls
	public function set_controls() {
		$this->controls['info'] = [
			'tab' 		=> 'content',
			'type' 		=> 'info',
			'content' 	=> sprintf( __( 'A <a href="%s" target="_blank">free shipping method</a> must be enabled in your shipping zone and be set to the "Free Shipping Requires", "Minimum order amount", etc setting.', 'woocommerce' ) . '<br/><br/>Shortcodes:<br/>1. ##min_amount##<br/>2. ##cart_total##<br/>3. ##update_min_amount##<br/><br/>$200 is a placeholder value for builder editor.', 'https://docs.woocommerce.com/document/free-shipping/' )
		];

		$this->controls['message'] = [
			'tab' 		=> 'content',
			'type' 		=> 'textarea',
			'label' 	=> esc_html__( 'Free Shipping Notice', 'bricksultimate' ),
			'placeholder' => __( 'You\'re ##update_min_amount## away from free shipping.', 'bricksultimate' ),
		];

		$this->controls['action'] = [
			'tab' 		=> 'content',
			'type' 		=> 'select',
			'label' 	=> esc_html__( 'Action', 'bricksultimate' ),
			'info' 		=> esc_html__( 'When free shipping amount is reached.', "bricksultimate" ),
			'options' 	=> [
				'hide' 	=> __('Hide'),
				'msg' 	=> __('Show success message', 'bricksultimate')
			]
		];

		$this->controls['success_msg'] = [
			'tab' 		=> 'content',
			'type' 		=> 'textarea',
			'label' 	=> esc_html__( 'Success Message', 'bricksultimate' ),
			'info' 		=> esc_html__( 'When free shipping amount is reached.', "bricksultimate" ),
			'required'  => ['action', '=', 'msg']
		];

		$this->controls['thresholdAmount'] = [
			'tab' 		=> 'content',
			'label' 	=> esc_html__( 'Threshold amount', 'bricksultimate' ),
			'info' 		=> esc_html__( 'Threshold amount after which notice should start appear. Do not enter the currency.', "bricksultimate" ),
			'type' 		=> 'text',
			'placeholder' 	=> 50,
			'hasDynamicData' => false
		];

		$this->controls['textTypography'] = [
			'tab' 		=> 'content',
			'label' 	=> esc_html__( 'Text Typography', 'bricksultimate' ),
			'type' 		=> 'typography',
			'css' 		=> [
				[
					'property' => 'font',
				],
			]
		];

		$this->controls['priceTypography'] = [
			'tab' 		=> 'content',
			'label' 	=> esc_html__( 'Price Typography', 'bricksultimate' ),
			'type' 		=> 'typography',
			'css' 		=> [
				[
					'property' => 'font',
					'selector' => '.woocommerce-Price-amount.amount',
				],
			]
		];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_script( 'bu-fsn-text', Helpers::get_asset_url('js') . 'fsntext.min.js', ['wc-cart-fragments'], filemtime(Helpers::get_asset_path('js') . 'fsntext.min.js'), true );
	}

	// Render element HTML
	public function render() {
		$settings = $this->settings;

		$min_amount = WooCommerce::freeShippingMinAmount();
		$decimals 	= \wc_get_price_decimals() == 0 ? 2 : \wc_get_price_decimals();
		$action 	= Helpers::get_value( $settings, 'action', 'hide');

		if( Helpers::isBricksBuilderActive() && Helpers::parseCurrency( $min_amount ) <= 0 )
			$min_amount = number_format( 200, $decimals, wc_get_price_decimal_separator(), wc_get_price_thousand_separator() );

		$thresholdAmount = Helpers::get_value( $settings, 'thresholdAmount', false);
		$formattedTotal = number_format( WooCommerce::get_customer_order_amount(), $decimals, wc_get_price_decimal_separator(), wc_get_price_thousand_separator() );

		$success_msg = Helpers::get_value( $settings, 'success_msg', false);

		$message = Helpers::get_value( $settings, 'message', __( 'You\'re ##update_min_amount## away from free shipping.', 'bricksultimate' ) );
		$message = str_replace( "##min_amount##", wc_price( $min_amount ), $message );
		$message = str_replace( "##cart_total##", sprintf( "<span class=\"bu-cart-subtotal\">%s</span>", WooCommerce::calculate_cart_subtotal() ), $message );
		$message = str_replace( "##update_min_amount##", sprintf( "<span class=\"fsn-updated-price\">%s</span>", wc_price( WooCommerce::freeShippingUpdatedAmount() ) ), $message );

		if( $thresholdAmount && ( Helpers::parseCurrency( $thresholdAmount ) > Helpers::parseCurrency( $formattedTotal ) ) && ! Helpers::isBricksBuilderActive() ) {
			$this->set_attribute('_root', 'class', 'screen-reader-text');
		}

		if( ( WooCommerce::hasFreeShippingMethod() === 'no' || ( $action == 'hide' && Helpers::parseCurrency( $formattedTotal ) > Helpers::parseCurrency( $min_amount ) ) ) && ! Helpers::isBricksBuilderActive() ) {
			$this->set_attribute('_root', 'class', 'screen-reader-text');
		}

		if( ( $action == 'msg' && Helpers::parseCurrency( $formattedTotal ) > Helpers::parseCurrency( $min_amount ) ) && ! Helpers::isBricksBuilderActive() ) {
			$this->set_attribute('bu_fsn_text_notice', 'class', 'screen-reader-text');
		}

		$attr = [
			'minAmount' 		=> $min_amount,
			'thresholdAmount' 	=> ( $thresholdAmount ) ? number_format( $thresholdAmount, $decimals, wc_get_price_decimal_separator(), wc_get_price_thousand_separator() ) : 'no',
			'action' 			=> $action
		];

		$this->set_attribute('_root', 'class', 'bu-fsn-text');
		$this->set_attribute( '_root', 'data-fsn-text', wp_json_encode( $attr ) );

		echo "<{$this->tag} {$this->render_attributes('_root')}>";

		$this->set_attribute('bu_fsn_text_notice', 'class', 'bu-fsn-text--notice');
		echo "<span {$this->render_attributes('bu_fsn_text_notice')}>{$message}</span>";

		if( $success_msg ) {
			if( ( $action == 'msg' && Helpers::parseCurrency( $formattedTotal ) > Helpers::parseCurrency( $min_amount ) ) && ! Helpers::isBricksBuilderActive() ) {
				$this->set_attribute('bu_fsn_text_sucmsg', 'class', 'bu-fsn-text--success');
			} else {
				$this->set_attribute('bu_fsn_text_sucmsg', 'class', 'bu-fsn-text--success screen-reader-text');
			}
			echo "<span {$this->render_attributes('bu_fsn_text_sucmsg')}>{$success_msg}</span>";
		}

		echo "</{$this->tag}>";
	}
}