<?php
namespace BricksUltimate\Elements;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use BricksUltimate\Helpers;
use BricksUltimate\Trait\TooltipControls;

class WooSwatchesLoop extends \Bricks\Element {
	use TooltipControls;

	public $category = 'ultimate';
	public $name 	= 'bu-swatches-loop';
	public $icon 	= 'ti-palette';

	// Return localized element label
	public function get_label() {
		return esc_html__( '(BU)Swatches for Loop', 'bricksultimate' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['_clearBtn'] = array(
			'title' 	=> esc_html__( 'Clear Text', 'bricksultimate' ),
			'tab'   	=> 'content',
			'required'  => ['enableSync', '=', true]
		);

		$this->control_groups['tooltip'] = [
			'title' 	=> esc_html__( 'Tooltip', 'bricksultimate' ),
			'tab' 		=> 'content',
		];
	}

	// Set builder controls
	public function set_controls() {
		$attributes = ['none' => esc_html__( 'Select attribute', 'bricksultimate' ) ];
		$wc_attributes = wc_get_attribute_taxonomies();
		if( $wc_attributes ) {
			foreach( $wc_attributes as $attribute ) {
				$attributes[ 'pa_' . $attribute->attribute_name] = ucwords( $attribute->attribute_label );
			}
		}

		$this->controls['swTextSwatches'] = [
			'type' 		=> 'checkbox',
			'label' 	=> esc_html__( 'Text only(no images or color)', 'bricksultimate' ),
		];

		$this->controls['swDisableVariationImage'] = [
			'type' 		=> 'checkbox',
			'label' 	=> esc_html__( "No image from the variations", 'bricksultimate' ),
			'description' => esc_html__( "Don't display the variations image.", 'bricksultimate' ),
		];

		$this->controls['syncatc'] = [
			'type' => 'separator',
			'label' => esc_html__('Sync with Add To Cart Button')
		];
		
		$this->controls['enableSync'] = [
			'type' 	=> 'checkbox',
			'label' => esc_html__('Enable', 'bricksultimate'),
			'description' => esc_html__( 'Do you want to sync the swatches with the add-to-cart button? It would only work for the variable product type.', 'bricksultimate' ),
		];
		
		$this->controls['buttonElement'] = [
			'type' 	=> 'select',
			'label' => esc_html__('Button element', 'bricksultimate'),
			'options' => [
				'atc' 	=> esc_html__( 'Add To Cart', 'bricksultimate' ),
				'atci' 	=> esc_html__( '(BU) Add To Cart Icon', 'bricksultimate' ),
			],
			'default' => 'atc',
			'placeholder' => esc_html__( 'Add To Cart', 'bricksultimate' ),
			'required' => ['enableSync', '=', true]
		];
		
		$this->controls['variableButtonText'] = [
			'type' 	=> 'text',
			'label' => esc_html__('Select options button text', 'bricksultimate'),
			'placeholder' => esc_html__('Add To Cart', 'bricksultimate'),
			'description' => esc_html__( 'Replacing the "Select Options" button text when user selects the swatches.', 'bricksultimate' ),
			'required' => [ ['enableSync', '=', true], ['buttonElement', '=', 'atc'] ]
		];

		$this->controls['syncatcsepend'] = [
			'type' => 'separator',
		];

		$this->controls['swAttribute'] = [
			'type'        => 'select',
			'label'       => esc_html__( 'Select an attribute', 'woocommerce' ),
			'options'     => $attributes,
			'placeholder' => esc_html__( 'Select attribute', 'bricksultimate' ),
			'required' => ['enableSync', '!=', true]
		];

		$this->controls['swLimit'] = [
			'type' 		=> 'number',
			'units' 	=> false,
			'label' 	=> esc_html__( 'Swatches limit', 'bricksultimate' ),
			'placeholder' => 3,
			'required' => ['enableSync', '!=', true]
		];

		$this->controls['swLimitButton'] = [
			'type' 		=> 'checkbox',
			'label' 	=> esc_html__( 'Show limit button', 'bricksultimate' ),
			'rerender' 	=> false,
			'required' => ['enableSync', '!=', true]
		];

		$this->controls['swForSimpleProduct'] = [
			'type' 		=> 'checkbox',
			'label' 	=> esc_html__( 'Enable for simple product', 'bricksultimate' ),
			'description' => esc_html__( 'Displays the swatches for the simple product type.', 'bricksultimate' ),
			'rerender' 	=> false,
		];

		$this->controls['swDisableImageUpdate'] = [
			'type' 		=> 'checkbox',
			'label' 	=> esc_html__( 'Disable selected variation image', 'bricksultimate' ),
			'rerender' 	=> false
		];

		$this->controls['swMainImage'] = [
			'type' 		=> 'text',
			'label' 	=> esc_html__( 'Featured image selector', 'bricksultimate' ),
			'placeholder' => '.brxe-image > a > img',
			'description' => esc_html__( 'Display the selected variations image.', 'bricksultimate' ),
			'rerender' 	=> false,
			'required' => ['swDisableImageUpdate', '!=', true]
		];

		$this->controls['swPriceSelector'] = [
			'type' 	=> 'text',
			'label' => esc_html__( 'Price element selector', 'bricksultimate' ),
			'description' => esc_html__('Show the price of the selected variations', 'bricksultimate'),
			'placeholder' => '.brxe-product-price',
			'required' => ['enableSync', '=', true]
		];

		$this->controls['swSep'] = [
			'type' 	=> 'separator',
		];

		$this->controls['swHideLabel'] = [
			'type' 	=> 'checkbox',
			'label' => esc_html__( 'Hide label', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'display',
					'selector' => 'th.label',
					'value' 	=> 'none'
				]
			],
			'required' => ['enableSync', '=', true]
		];

		$this->controls['swLabelTypography'] = [
			'type' 	=> 'typography',
			'label' => esc_html__( 'Label typography', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'font',
					'selector' => 'th.label label, th.label',
				]
			],
			'required' => [ ['enableSync', '=', true], ['swHideLabel', '!=', true]]
		];

		$this->controls['swAlign'] = [
			'type' 	=> 'text-align',
			'label' => esc_html__( 'Content alignment', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'text-align',
					'selector' => 'table.variations td',
				]
			],
			'required' => [ ['enableSync', '=', true], ['swHideLabel', '=', true]]
		];

		$this->controls['swGap'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Gap', 'bricksultimate' ),
			'units' => true,
			'placeholder' => '4px',
			'css' 	=> [
				[
					'property' => 'margin-right',
					'selector' => '.swatch-on-loop:not(:last-child)'
				]
			]
		];

		$this->controls['swOuterBrd'] = [
			'type' 	=> 'border',
			'label' => esc_html__( 'Outer border', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'border',
					'selector' => '.bu-swatch'
				]
			],
			'exclude' => ['border-radius']
		];

		$this->controls['swDisable'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Opacity - disabled swatches', 'bricksultimate' ),
			'unitless' => true,
			'min' 	=> 0,
			'max' 	=> 1,
			'step' => 0.1,
			'placeholder' => 0.75,
			'css' 	=> [
				[
					'property' => 'opacity',
					'selector' => '.swatch-disabled, .swatch-ofs'
				]
			]
		];

		$this->controls['swLimitTg'] = [
			'type' 	=> 'typography',
			'label' => esc_html__( 'Limit button typography', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'font',
					'selector' => '.bu-swatches-divider'
				]
			],
			'exclude' => ['letter-spacing', 'text-decoration', 'text-transform', 'text-align', 'font-style'],
			'required' => ['swLimitButton', '=', true]
		];

		$this->controls['swColorSep'] = [
			'type' 	=> 'separator',
			'label' => esc_html__( 'Color swatches', 'bricksultimate' ),
		];

		$this->controls['swColorW'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Width', 'bricksultimate' ),
			'units' => true,
			'placeholder' => '18px',
			'css' 	=> [
				[
					'property' => 'width',
					'selector' => '.swatch-with-bg.swatches-color span'
				]
			]
		];

		$this->controls['swColorH'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Height', 'bricksultimate' ),
			'units' => true,
			'placeholder' => '18px',
			'css' 	=> [
				[
					'property' => 'height',
					'selector' => '.swatch-with-bg.swatches-color span'
				]
			]
		];

		$this->controls['swColorbrd'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Border radius', 'bricksultimate' ),
			'units' => true,
			'css' 	=> [
				[
					'property' => 'border-radius',
					'selector' => '.swatch-with-bg.swatches-color, .swatch-with-bg.swatches-color span'
				]
			]
		];

		$this->controls['swWhiteTick'] = [
			'type' 	=> 'checkbox',
			'label' => esc_html__( 'Disable white tick icon', 'bricksultimate' )
		];

		$this->controls['swImgSep'] = [
			'type' 	=> 'separator',
			'label' => esc_html__( 'Image swatches', 'bricksultimate' ),
		];

		$this->controls['swImgW'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Width', 'bricksultimate' ),
			'units' => true,
			'placeholder' => '25px',
			'css' 	=> [
				[
					'property' => 'width',
					'selector' => '.swatch-with-bg.swatches-image span'
				]
			]
		];

		$this->controls['swImgH'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Height', 'bricksultimate' ),
			'units' => true,
			'placeholder' => '25px',
			'css' 	=> [
				[
					'property' => 'height',
					'selector' => '.swatch-with-bg.swatches-image span'
				]
			]
		];

		$this->controls['swImgbrd'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Border radius', 'bricksultimate' ),
			'units' => true,
			'placeholder' => '0px',
			'css' 	=> [
				[
					'property' => 'border-radius',
					'selector' => '.swatch-with-bg.swatches-image, .swatch-with-bg.swatches-image span'
				]
			]
		];

		$this->controls['swTextSep'] = [
			'type' 	=> 'separator',
			'label' => esc_html__( 'Text swatches', 'bricksultimate' ),
		];

		$selector = '.bu-swatch:not(.swatch-with-bg) span';

		$this->controls['swTextW'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Width', 'bricksultimate' ),
			'units' => true,
			'css' 	=> [
				[
					'property' => 'width',
					'selector' => $selector
				]
			]
		];

		$this->controls['swTextH'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Height', 'bricksultimate' ),
			'units' => true,
			'css' 	=> [
				[
					'property' => 'height',
					'selector' => $selector
				]
			]
		];

		$this->controls['swTextFont'] = [
			'type' 	=> 'typography',
			'label' => esc_html__( 'Typography', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'font',
					'selector' => $selector
				]
			]
		];

		$this->controls['swTextBg'] = [
			'type' 	=> 'color',
			'label' => esc_html__( 'Background', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'background-color',
					'selector' => $selector
				]
			]
		];

		$this->controls['swTextbrd'] = [
			'type' 	=> 'number',
			'label' => esc_html__( 'Border radius', 'bricksultimate' ),
			'units' => true,
			'css' 	=> [
				[
					'property' => 'border-radius',
					'selector' => $selector
				]
			]
		];

		$this->controls['swTextPad'] = [
			'type' 	=> 'dimensions',
			'label' => esc_html__( 'Padding', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'padding',
					'selector' => $selector
				]
			]
		];

		$this->controls['swActiveSep'] = [
			'type' 	=> 'separator',
			'label' => esc_html__( 'Active swatches', 'bricksultimate' ),
		];

		$this->controls['swTextABg'] = [
			'type' 	=> 'color',
			'label' => esc_html__( 'Background', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'background-color',
					'selector' => '.active-swatch.bu-swatch:not(.swatch-with-bg) span'
				]
			]
		];

		$this->controls['swTextAClr'] = [
			'type' 	=> 'color',
			'label' => esc_html__( 'Text color', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'color',
					'selector' => '.active-swatch.bu-swatch:not(.swatch-with-bg) span'
				]
			]
		];

		$this->controls['swActiveBrd'] = [
			'type' 	=> 'border',
			'label' => esc_html__( 'Border', 'bricksultimate' ),
			'css' 	=> [
				[
					'property' => 'border',
					'selector' => '.bu-swatch.active-swatch'
				]
			],
			'exclude' => ['border-radius']
		];

		/**
		 * Clear text button
		 ----------------------------------------------------------------*/
		$selector = 'form.variations_form table.variations .reset_variations';

		$this->controls['_clrBtnHide'] = [
			'group' => '_clearBtn',
			'label' => esc_html__( 'Remove the button?', 'bricks' ),
			'type'  => 'checkbox',
			'css'   => [
				[
					'property' 	=> 'display',
					'selector' 	=> '.sw-reset-button',
					'value' 	=> 'none',
					'important' => true
				],
			]
		];

		$this->controls['_clrBtnText'] = [
			'group' => '_clearBtn',
			'label' => esc_html__( 'Text', 'bricks' ),
			'type'  => 'text',
			'default' => esc_html__('Clear', 'bricksultimate')
		];

		$this->controls['_clrBtnWidth'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'label' => esc_html__( 'Width', 'bricks' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'width',
					'selector' 	=> $selector,
				],
			]
		];

		$this->controls['_clrBtnHeight'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'label' => esc_html__( 'Height', 'bricks' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'height',
					'selector' 	=> $selector,
				],
			],
			'info'  => __( 'Set to "100vh" for full height.', 'bricks' ),
		];

		$this->controls['_clrBtnSpacingSep'] = [
			'tab'   	=> 'content',
			'group' 	=> '_clearBtn',
			'label' 	=> esc_html__( 'Spacing', 'bricks' ),
			'type' 		=> 'separator',
		];

		$this->controls['_clrBtnMargin'] = [
			'tab'   	=> 'content',
			'group' 	=> '_clearBtn',
			'label' 	=> esc_html__( 'Margin', 'bricks' ),
			'type'  	=> 'dimensions',
			'css'   => [
				[
					'property' => 'margin',
					'selector' 	=> $selector,
				],
			]
		];

		$this->controls['_clrBtnPadding'] = [
			'tab'   	=> 'content',
			'group' 	=> '_clearBtn',
			'label' 	=> esc_html__( 'Padding', 'bricks' ),
			'type'  	=> 'dimensions',
			'css'   => [
				[
					'property' => 'padding',
					'selector' 	=> $selector,
				],
			]
		];

		$this->controls['_clrBtnStyleSep'] = [
			'tab'   	=> 'content',
			'group' 	=> '_clearBtn',
			'type' 		=> 'separator',
		];

		$this->controls['_clrBtnBgColor'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'type'  => 'color',
			'label' => esc_html__('Background', 'bricks'),
			'css' 	=> [
				[
					'selector' 	=> $selector,
					'property' => 'background-color'
				],
			]
		];

		$this->controls['_clrBtnTypography'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'type'  => 'typography',
			'label' => esc_html__('Typography', 'bricks'),
			'css' 	=> [
				[
					'selector' 	=> $selector,
					'property' => 'font'
				],
			],
		];

		$this->controls['_clrBtnBorder'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'type'  => 'border',
			'label' => esc_html__('Border', 'bricks'),
			'css' 	=> [
				[
					'selector' 	=> $selector,
					'property' => 'border'
				],
			],
		];

		$this->controls['_clrBtnBoxshadow'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'type'  => 'box-shadow',
			'label' => esc_html__('Box shadow', 'bricks'),
			'css' 	=> [
				[
					'selector' 	=> $selector,
					'property' => 'box-shadow',
				],
			],
		];

		$this->controls['_clrBtnHoverSep'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'type'  => 'separator',
			'label' => esc_html__('Hover State', 'bricks'),
		];

		$this->controls['_clrBtnHBgColor'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'type'  => 'color',
			'label' => esc_html__('Background', 'bricks'),
			'css' 	=> [
				[
					'selector' => $selector . ':hover',
					'property' => 'background-color'
				],
			],
		];

		$this->controls['_clrBtnHoverColor'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'type'  => 'color',
			'label' => esc_html__('Color', 'bricks'),
			'css' 	=> [
				[
					'selector' => $selector . ':hover',
					'property' => 'color'
				],
			],
		];

		$this->controls['_clrBtnHBorder'] = [
			'tab'   => 'content',
			'group' => '_clearBtn',
			'type'  => 'border',
			'label' => esc_html__('Border', 'bricks'),
			'css' 	=> [
				[
					'selector' => $selector . ':hover',
					'property' => 'border'
				],
			],
		];

		$this->controls['_clrBtnHBoxshadow'] = [
			'group' => '_clearBtn',
			'type'  => 'box-shadow',
			'label' => esc_html__('Box shadow', 'bricks'),
			'css' 	=> [
				[
					'selector' => $selector . ':hover',
					'property' => 'box-shadow',
				],
			],
		];

		$this->controls['_clrBtnIconSep'] = [
			'group' => '_clearBtn',
			'label' => esc_html__( 'Icon', 'bricks' ),
			'type'  => 'separator',
		];

		$this->controls['_clrBtnIcon'] = [
			'group' => '_clearBtn',
			'label' => esc_html__( 'Icon', 'bricks' ),
			'type'  => 'icon',
		];

		$this->controls['_clrBtnIconGap'] = [
			'group' => '_clearBtn',
			'label' => esc_html__( 'Gap', 'bricks' ),
			'type'  => 'number',
			'units' => true,
			'css' 	=> [
				[
					'property' => 'margin-left',
					'selector' => '.reset-btn-icon'
				]
			]
		];

		$this->controls['_clrBtnIconSize'] = [
			'group' => '_clearBtn',
			'label' => esc_html__( 'Size', 'bricks' ),
			'type'  => 'number',
			'units' => true,
			'css' 	=> [
				[
					'property' => 'font-size',
					'selector' => '.reset-btn-icon'
				]
			]
		];

		$this->controls['_clrBtnIconColor'] = [
			'group' => '_clearBtn',
			'label' => esc_html__( 'Color', 'bricks' ),
			'type'  => 'color',
			'css' 	=> [
				[
					'property' => 'color',
					'selector' => '.reset-btn-icon'
				],
				[
					'property' => 'fill',
					'selector' => '.reset-btn-icon'
				]
			]
		];

		$this->controls['_clrBtnIconHColor'] = [
			'group' => '_clearBtn',
			'label' => esc_html__( 'Hover color', 'bricks' ),
			'type'  => 'color',
			'css' 	=> [
				[
					'property' => 'color',
					'selector' => '.reset_variations:hover .reset-btn-icon'
				],
				[
					'property' => 'fill',
					'selector' => '.reset_variations:hover .reset-btn-icon'
				]
			]
		];

		/**
		 * Tooltip
		-----------------------------------------------*/
		$this->controls = $this->bu_tooltip_controls( $this->controls, 'tooltip' );
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_script( 'bu-swatches' );

		if( isset($this->settings['hasTooltip']) ) {
			wp_enqueue_style( 'bu-tooltip' );
			wp_enqueue_script( 'bu-popper' );
			wp_enqueue_script( 'bu-tippy' );
		}

		if( ! empty($this->settings['enableSync']) ) {
			// Enqueue variation scripts.
			wp_enqueue_script( 'wc-add-to-cart-variation' );
		}
	}

	// Render element HTML
	public function render() {
		$settings = $this->settings;

		global $product;

		$product = wc_get_product( $this->post_id );

		if( $product == false && ( bricks_is_builder() || bricks_is_builder_iframe() || Helpers::isBricksBuilderActive() ) ) {
			$product = wc_get_products(
				[
					'limit'   => 1,
					'orderby' => 'date',
					'order'   => 'DESC',
					'tax_query'      => array(
						'relation' => 'AND',
						array(
							'taxonomy' => 'product_type',
							'field'    => 'slug',
							'terms'    => array( 'variable' ),
						)
					),
				]
			)[0];
		}

		if ( empty( $product ) ) {
			return $this->render_element_placeholder(
				[
					'title'       => esc_html__( 'For better preview select content to show.', 'bricks' ),
					'description' => esc_html__( 'Go to: Settings > Template Settings > Populate Content', 'bricks' ),
				]
			);
		}
		
		$attribute = Helpers::get_value($settings, 'swAttribute', 'none');

		if ( ( empty( $attribute ) || $attribute == false || $attribute == 'none' ) && empty( $settings['enableSync'] ) ) {
			return $this->render_element_placeholder(
				[
					'title' => esc_html__( 'Select the attribute', 'bricks' ),
				]
			);
		}
		
		if( $product->has_attributes() && $product->is_type( 'simple' ) && ! empty( $settings['swForSimpleProduct'] ) ) {
			// Get the product attributes
			$attributes = $product->get_attributes();
			$terms = $swatches = [];
			
			foreach ($attributes as $attribute) {
				if ($attribute->is_taxonomy()) {
					$taxonomy = $attribute->get_name();
					$terms = wp_get_post_terms($product->get_id(), $taxonomy, array('fields' => 'ids'));
				} else {
					$attribute_name = $attribute->get_name();
					$terms = $attribute->get_options();
				}
			}
			
			foreach( $terms as $term_id ) {
				$color 		= get_term_meta( $term_id, 'bu_swatches_color', true );
				$bbcolor 	= get_term_meta( $term_id, 'bricks_swatch_color_value', true );
				$image_id 	= get_term_meta( $term_id, 'bu_swatches_img_id', true );
				$bbimg_id 	= get_term_meta( $term_id, 'bricks_swatch_image_value', true );

				if ( $color != '' ) {
					$swatches[$term_id]['color'] = $color;
				} elseif( $bbcolor != '' ) {
					$swatches[$term_id]['color'] = esc_attr( $bbcolor );
				}

				if ( $image_id != '' ) {
					$swatches[$term_id]['image'] = wp_get_attachment_image_url( $image_id, 'full' );
				} elseif( $bbimg_id != '' ) {
					$swatches[$term_id]['image'] = wp_get_attachment_image_url( $image_id, 'full' );
				}
				
				$label_text = get_term_meta( $term_id, 'bricks_swatch_label_value', true );
				$swatches[$term_id]['org'] = ( $label_text != '') ? esc_attr( $label_text ) : get_term( $term_id )->name;
			}
			
			if( ! empty( $settings['hasTooltip'] ) ) {
				$data = [
					'hasTooltip' => 'yes',
					'animation' => Helpers::get_value($settings, 'ttAnimation', 'perspective'),
					'placement' => Helpers::get_value($settings, 'ttPlacement', 'top')
				];

				$this->set_attribute( '_root',  "data-tooltip-config", wp_json_encode( $data ) );
			}

			if( ! empty( $settings['swMainImage'] ) ) {
				$this->set_attribute('_root', 'data-featimg-sel', esc_attr( $settings['swMainImage'] ) );
			}

			if( ! empty( $settings['swDisableImageUpdate'] ) )
				$this->set_attribute('_root', 'class', ['bu-no-image-update']);

			$this->set_attribute('_root', 'class', ['swatches-select']);

			if( ! empty( $settings['swWhiteTick'] ) ){
				$this->set_attribute( '_root', 'class', [ 'no-tick' ] );
			}

			$output = "<{$this->tag} {$this->render_attributes('_root')}>";
			$loopIndex = 0;
			
			foreach ( $swatches as $key => $swatch ) {
				$class = $style = $data = '';

				$swatchLimit = Helpers::get_value($settings, 'swLimit', 99 );
				if ( count( $swatches ) > $swatchLimit ) {
					if ( $loopIndex >= $swatchLimit ) {
						$class .= ' bu-swatch-hidden';
					}

					if ( $loopIndex == $swatchLimit && !empty( $settings['swLimitButton'] ) ) {
						$output .= '<div class="bu-swatches-divider">+' . ( count( $swatches ) - $swatchLimit ) . '</div>';
					}
				}

				$loopIndex++;

				if ( ! empty( $swatch['color'] ) && empty( $settings['swTextSwatches'] ) ) {
					$style = 'background-color:' . $swatch['color'];
					$class .= ' swatch-with-bg swatches-color';
				} elseif ( ! empty( $swatch['image'] ) && empty( $settings['swTextSwatches'] ) ) {
					$style = 'background-image: url(' . $swatch['image'] . ')';
					$class .= ' swatch-with-bg swatches-image';
				} else {
					$class .= ' text-only ';
				}

				$data .= ' data-title="' . esc_attr( $swatch['org'] ) . '"';
				$output .= '<div class="swatch-on-loop bu-swatch' . esc_attr( $class ) . '" ' . $data . '><span style="' . esc_attr( $style ) . '">' . $swatch['org'] . '</span></div>';
			}
			
			$output .= "</{$this->tag}>";

			echo $output;
			
			return;
		}

		//******Variable products
		if ( ! $product->is_type( 'variable' ) ) {
			return false;
		}

		add_filter('woocommerce_available_variation', [$this, 'bu_filter_varation_args'], 10, 3);

		$variations = $product->get_available_variations();

		remove_filter('woocommerce_available_variation', [$this, 'bu_filter_varation_args'], 10, 3);
		
		if ( empty( $variations ) ) {
			return false;
		}

		$attributes = $product->get_variation_attributes();

		if( ! empty( $settings['hasTooltip'] ) ) {
			$data = [
				'hasTooltip' => 'yes',
				'animation' => Helpers::get_value($settings, 'ttAnimation', 'perspective'),
				'placement' => Helpers::get_value($settings, 'ttPlacement', 'top')
			];

			$this->set_attribute( '_root',  "data-tooltip-config", wp_json_encode( $data ) );
		}

		if( ! empty( $settings['swMainImage'] ) ) {
			$this->set_attribute('_root', 'data-featimg-sel', esc_attr( $settings['swMainImage'] ) );
		}

		if( ! empty( $settings['swDisableImageUpdate'] ) )
			$this->set_attribute('_root', 'class', ['bu-no-image-update']);

		$this->set_attribute('_root', 'class', ['swatches-select']);

		if( ! empty( $settings['swWhiteTick'] ) ){
			$this->set_attribute( '_root', 'class', [ 'no-tick' ] );
		}

		if( ! empty( $settings['enableSync'] ) && ! empty( $settings['swPriceSelector'] ) ) {
			$updateprice = [
				'updatePrice' => 'yes',
				'selector' => Helpers::get_value($settings, 'swPriceSelector', false)
			];

			$this->set_attribute( '_root',  "data-price-update", wp_json_encode( $updateprice ) );
		}

		if ( \Bricks\Database::get_setting( 'woocommerceUseVariationSwatches' ) ) {
			remove_filter( 'woocommerce_dropdown_variation_attribute_options_html', [ new \Bricks\Product_Add_To_Cart(), 'maybe_convert_to_swatches' ], 100, 2 );
		}

		$output = "<{$this->tag} {$this->render_attributes('_root')}>";

		if( ! empty( $settings['enableSync'] ) ) {
			// Get Available variations?
			$get_variations = count( $product->get_children() ) <= apply_filters( 'woocommerce_ajax_variation_threshold', 30, $product );
			$path = \BricksUltimate\Plugin::instance()->get_root_path() . 'templates/';
			ob_start();

			// Load the template.
			\wc_get_template(
				'single-product/add-to-cart/swatches-for-loop.php',
				array(
					'available_variations' => $get_variations ? $variations : false,
					'attributes'           => $attributes,
					'selected_attributes'  => $product->get_default_attributes(),
					'settings' 				=> $settings,
					'path' 					=> $path
				),
				$path,
				$path
			);

			$output .= ob_get_clean();
		} else {
			$swatches = \BricksUltimate\WooCommerce\Swatches::bu_get_product_variations_images( $attribute, $variations, false, $this->post_id );

			if ( empty( $swatches ) ) {
				return false;
			}

			$loopIndex = 0;
			foreach ( $swatches as $key => $swatch ) {
				$class = $style = $data = '';
				
				$swatchLimit = Helpers::get_value($settings, 'swLimit', 99 );
				if ( count( $swatches ) > $swatchLimit ) {
					if ( $loopIndex >= $swatchLimit ) {
						$class .= ' bu-swatch-hidden';
					}
					
					if ( $loopIndex == $swatchLimit && !empty( $settings['swLimitButton'] ) ) {
						$output .= '<div class="bu-swatches-divider">+' . ( count( $swatches ) - $swatchLimit ) . '</div>';
					}
				}
				
				$loopIndex++;

				if ( ! empty( $swatch['color'] ) && empty( $settings['swTextSwatches'] ) ) {
					$style = 'background-color:' . $swatch['color'];
					$class .= ' swatch-with-bg swatches-color';
				} elseif ( isset( $swatch['image_src'] ) && empty( $settings['swTextSwatches'] ) && empty( $settings['swDisableVariationImage'] ) ) {
					$thumb = wp_get_attachment_image_src( get_post_thumbnail_id( $swatch['variation_id'] ), 'woocommerce_thumbnail' );
					if ( ! empty( $thumb ) ) {
						$style  = 'background-image: url(' . $thumb[0] . ')';
						$class .= ' swatch-with-bg swatches-image';
					}
				} elseif ( ! empty( $swatch['image'] ) && empty( $settings['swTextSwatches'] ) ) {
					$style = 'background-image: url(' . $swatch['image'] . ')';
					$class .= ' swatch-with-bg swatches-image';
				} else {
					$class .= ' text-only ';
				}

				if ( isset( $swatch['image_src'] ) ) {
					$data .= 'data-image-src="' . $swatch['image_src'] . '" ';
					$data .= 'data-image-srcset="' . $swatch['image_srcset'] . '" ';
					$data .= 'data-image-sizes="' . $swatch['image_sizes'] . '"';

					if ( ! $swatch['is_in_stock'] ) {
						$class .= ' out-of-stock';
					}
				}

				$variation_id = $swatch['variation_id'] ?? false;
				if( $variation_id && count( $attributes ) < 2 ) {
					$variation = wc_get_product( $variation_id );
					if( $variation && $variation->get_manage_stock() && ! $swatch['is_in_stock'] ) {
						$class .= ' swatch-ofs';
					}
				}

				$term = get_term_by( 'slug', $key, $attribute );
				if($term) {
					$data .= ' data-title="' . esc_attr( $term->name ) . '"';
					$output .= '<div class="swatch-on-loop bu-swatch' . esc_attr( $class ) . '" ' . $data . '><span style="' . esc_attr( $style ) . '">' . $term->name . '</span></div>';
				}
			}

		}

		$output .= "</{$this->tag}>";

		echo $output;
	}

	public function bu_filter_varation_args( $args, $object, $variation ) {
		unset( $args['backorders_allowed'] );
		unset( $args['dimensions'] );
		unset( $args['dimensions_html'] );
		unset( $args['is_downloadable'] );
		unset( $args['max_qty'] );
		unset( $args['min_qty'] );
		unset( $args['sku'] );
		unset( $args['variation_description'] );
		unset( $args['weight'] );
		unset( $args['weight_html'] );

		unset( $args['display_price'] );
		unset( $args['display_regular_price'] );

		return $args;
	}
}