<?php
namespace BricksUltimate;

if ( ! defined( 'ABSPATH' ) ) exit;

use BricksUltimate\Admin\{Admin, AdminMenuBar};
use BricksUltimate\{Helpers, Wishlist, Compare, Ajax};

class Plugin {
	/**
	 * BricksUltimate instance.
	 *
	 * Holds the plugin instance.
	 *
	 * @since  1.0.0
	 * @access public
	 * @static
	 *
	 * @var Plugin
	 */
	public static $instance = null;

	/**
	 * Plugin settings data from backend
	 *
	 * @var array
	 */
	public static $bu_settings = [];

	/**
	 * Gets all active elements
	 *
	 * @var array
	 */
	public static $general_elements = [];

	/**
	 * Gets all active woo elements
	 *
	 * @var array
	 */
	public static $woo_elements = [];

    /**
	 * Plugin data from header comments
	 *
	 * @var string
	 */
	private $version = null;

	/**
	 * Project root path
	 *
	 * @var string
	 */
	private $project_root_path = null;

	/**
	 * Project root url
	 *
	 * @var string
	 */
	private $project_root_url = null;

    /**
     * Holds a refference to the plugin data
     *
     * @var array
     */
	public $plugin_data = [];


    /**
     * Holds a refference to the plugin path
     *
     * @var string
     */
	public $plugin_file = null;

	/**
	 * Site URL
	 */
	const BU_SITE_URL = 'https://bricksultimate.com/';

    /**
     * Main class constructor
     *
     * @param string $path The plugin path.
     */
    public function __construct( $path ) {
        $this->plugin_file       = $path;
        $this->project_root_path = trailingslashit( dirname( $path ) );
        $this->project_root_url  = plugin_dir_url( $path );

        self::$instance = $this;
        
        add_action( 'plugins_loaded', [ $this, 'on_plugins_loaded' ], 100 );
    }

    /**
	 * Instance.
	 *
	 * Always load a single instance of the Plugin class
	 *
	 * @access public
	 * @static
	 *
	 * @return Plugin an instance of the class
	 */
	public static function instance() {
		return self::$instance;
	}

	/**
	 * Retrieve the project root path
	 *
	 * @return string
	 */
	public function get_root_path() {
		return $this->project_root_path;
	}

	/**
	 * Retrieve the project root path
	 *
	 * @return string
	 */
	public function get_plugin_file() {
		return $this->plugin_file;
	}


	/**
	 * Retrieve the project root url
	 *
	 * @return string
	 */
	public function get_root_url() {
		return $this->project_root_url;
	}

	/**
	 * Retrieve the project version
	 *
	 * @return string
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Retrieve the project data
	 *
	 * @param mixed $type
	 *
	 * @return string
	 */
	public function get_plugin_data( $type ) {
		if ( isset( $this->plugin_data[$type] ) ) {
			return $this->plugin_data[$type];
		}

		return null;
	}

	/**
	 * Will set the plugin data
	 *
	 * @param string $path
	 *
	 * @return array
	 */
	public function set_plugin_data( $path ) {
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		return get_plugin_data( $path );
	}

    /**
	 * Will load plugin helper methods
	 *
	 * @return void
	 */
	public function on_wp_init() {	
		if( ! function_exists('bricks_is_builder') )
    		return;

		if( version_compare( PHP_VERSION, '8.1.0', '<' ) ) {
			add_action( 'bricks_body', [ $this, 'bu_show_notification'] );
			return;
		}
		
		new Helpers();
	}

	public function bu_show_notification() {
		?>
		<div style="background-color: #900; color: #fff; text-align: center; width: 100%; padding: 8px 20px;">
			<p>
				<?php 
					_e("BricksUltimate addon needs at least PHP Version <strong>8.1</strong>, your version is ", 'bricksultimate'); echo '<strong>' . PHP_VERSION .'</strong>. ';
					_e('Please upgrade your PHP version.', 'bricksultimate');
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Init elements styles
	 */
	public function init_theme_styles() {
		if( version_compare( PHP_VERSION, '8.1.0', '<' ) ) {
			return;
		}

		$path = $this->get_root_path() . 'includes/ThemeStyles/';

		if( in_array( 'cf7', self::$general_elements ) ) {
			$file = $path . 'Cf7Styles.php';

			if ( is_readable( $file ) ) {
				require_once $file;

				new Cf7Styles();
			}
		}

		if( in_array( 'dualhd', self::$general_elements ) ) {
			$file = $path . 'DualColorTextStyles.php';

			if ( is_readable( $file ) ) {
				require_once $file;

				new DualColorTextStyles();
			}
		}

		if( in_array( 'ffs', self::$general_elements ) ) {
			$file = $path . 'FluentFormsStyles.php';

			if ( is_readable( $file ) ) {
				require_once $file;

				new FluentFormsStyles();
			}
		}

		if( in_array( 'gfs', self::$general_elements ) ) {
			$file = $path . 'GravityFormsStyles.php';

			if ( is_readable( $file ) ) {
				require_once $file;

				new GravityFormsStyles();
			}
		}

		if( in_array( 'wsfs', self::$general_elements ) ) {
			$file = $path . 'WSFormsStyler.php';

			if ( is_readable( $file ) ) {
				require_once $file;

				new WSFormsStyles();
			}
		}
	}

	/**
	 * Create the cookie
	 * 
	 * @param $cookie_key string cookie key
	 * @param $value string cookie value
	 * @param $time date cookie life
	 * 
	 * @return void
	 */
	public static function bu_setcookie( $cookie_key, $value, $time ) {
    	setcookie(
			$cookie_key,
			$value,
			[
				'expires'  => $time,
				'path'     => COOKIEPATH ? COOKIEPATH : '/',
				'domain'   => COOKIE_DOMAIN,
				'secure'   => is_ssl(),
				'httponly' => true,
				'samesite' => 'Strict',
			]
		);
    }

    /**
     * Will fire after the plugins are loaded and will initialize this plugin
     *
     * @return void
     */
    public function on_plugins_loaded() {
    	self::$bu_settings = get_option('bu_settings') ?? [];
		self::$general_elements = (array) get_option('bu_active_els') ?? [];
		self::$woo_elements = (array) get_option('bu_active_wcels') ?? [];
		
   		new Admin();
   		new Ajax();

   		add_action( 'init', 				[ $this, 'on_wp_init' ], 11 );
        add_action( 'init', 				[ $this, 'init_theme_styles' ], 9 );
   		add_action( 'after_setup_theme', 	[ $this, 'on_after_setup_theme' ] );
   		add_filter( 'all_plugins', 			[ $this, 'modify_plugin_branding' ] );
	}

	/**
	 * Setup loadtextdomain, HPOS and internal functions
	 * 
	 */
	public function on_after_setup_theme() {
		if( ! function_exists('bricks_is_builder') )
    		return;

    	load_plugin_textdomain( 'bricksultimate', false, $this->project_root_path . '/languages' );

    	$this->plugin_data 	= $this->set_plugin_data( $this->plugin_file );
       	$this->version 		= $this->plugin_data['Version'];

		add_action( 'before_woocommerce_init', function() {
			if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', $this->plugin_file, true );
			}
		});

		if( ! empty( self::$bu_settings['menu_bar'] ) && self::$bu_settings['menu_bar'] == 'enabled' )
			new AdminMenuBar();

		if( class_exists('WooCommerce') || in_array( 'wsl', self::$general_elements ) ) {
			new Wishlist();
		}

		if( class_exists('WooCommerce') || in_array( 'compare', self::$general_elements ) ) {
			new Compare();
		}

		if( class_exists('WooCommerce') ) {
			add_action( 'wp_enqueue_scripts', [ $this, 'add_inline_css' ] );

			if( function_exists( 'bricks_is_frontend' ) && bricks_is_frontend() ) {
				add_action('bricks_after_site_wrapper', function(){
					echo '<div id="buwoo-notice"></div>';
				}, 99);
			}
		}
	}

    public function add_inline_css() {
		wp_enqueue_style( 'bu-frontend', false);
		wp_add_inline_style( 
			'bricks-frontend', 
			"#buwoo-notice {cursor: pointer;position: fixed;right: 0px;z-index: 9999;bottom: 0px;margin-bottom: 15px; margin-right: 15px;} #buwoo-notice .notice-content{position: static; -webkit-transition: -webkit-transform 0.5s cubic-bezier(0.77, 0, 0.175, 1), box-shadow 0.5s cubic-bezier(0.77, 0, 0.175, 1); -o-transition: -o-transform 0.5s cubic-bezier(0.77, 0, 0.175, 1), box-shadow 0.5s cubic-bezier(0.77, 0, 0.175, 1); -moz-transition: -moz-transform 0.5s cubic-bezier(0.77, 0, 0.175, 1), box-shadow 0.5s cubic-bezier(0.77, 0, 0.175, 1); transition: transform 0.5s cubic-bezier(0.77, 0, 0.175, 1), box-shadow 0.5s cubic-bezier(0.77, 0, 0.175, 1); -webkit-transform: translate(110%, 0); -o-transform: translate(110%, 0); -moz-transform: translate(110%, 0); transform: translate(110%, 0);} .slide-from-right{-webkit-transform: none!important;-ms-transform: none !important;-moz-transform: none!important;transform: none!important;}" 
		);
	}

	/**
	 * Update plugin branding.
	 *
	 * @since 1.0
	 * @return array
	 */
	public function modify_plugin_branding( $all_plugins ) {
		$plugin_slug = plugin_basename( $this->plugin_file );
		if(! empty($all_plugins[$plugin_slug]) ) {
			$buwl = get_option('buwl');

			if( $buwl ) {
				$all_plugins[$plugin_slug]['Name'] 			= ! empty( $buwl['plugin_name'] ) ? esc_html( $buwl['plugin_name'] ) : $all_plugins[$plugin_slug]['Name'];
				$all_plugins[$plugin_slug]['PluginURI'] 	= ! empty( $buwl['plugin_uri'] ) ? esc_html( $buwl['plugin_uri'] ) : $all_plugins[$plugin_slug]['PluginURI'];
				$all_plugins[$plugin_slug]['Author'] 		= ! empty( $buwl['author_name'] ) ? esc_html( $buwl['author_name'] ) : $all_plugins[$plugin_slug]['Author'];
				$all_plugins[$plugin_slug]['AuthorURI'] 	= ! empty( $buwl['author_uri'] ) ? esc_html( $buwl['author_uri'] ) : $all_plugins[$plugin_slug]['AuthorURI'];
				$all_plugins[$plugin_slug]['Description'] 	= ! empty( $buwl['plugin_desc'] ) ? esc_html( $buwl['plugin_desc'] ) : $all_plugins[$plugin_slug]['Description'];
			}

			$all_plugins[$plugin_slug]['Title'] = $all_plugins[$plugin_slug]['Name'];
		}
		
		return $all_plugins;
	}
}