<?php
/**
 * Frames ACSS Connector Base class file.
 *
 * @package Frames_Client\ACSS_Connectors
 */

namespace Frames_Client\ACSS_Connectors;

use Frames_Client\Helpers\ACSS;
use Frames_Client\Helpers\Logger;

/**
 * ACSS_Connector_Base class.
 */
abstract class ACSS_Connector_Base {
	/**
	 * Array with Frames widgets to be injected in the ACSS dashboard
	 *
	 * @var array
	 */
	protected $widgets;

	/**
	 * List with default state for Frames Widgets
	 *
	 * @var array
	 */
	protected $default_widgets_states;

	/**
	 * Constructor method
	 *
	 * @param array $widgets List with the Frames Widgets.
	 * @param array $default_widgets_states List with default states for Frames Widgets.
	 * @return void
	 */
	public function __construct( array $widgets, array $default_widgets_states ) {
		$this->widgets = $widgets;
		$this->default_widgets_states = $default_widgets_states;
	}

	/**
	 * Connector initialization adding hooks for the proper version and loading the Frames widgets for Bricks
	 *
	 * @return void
	 */
	public function init(): void {
		$this->add_hooks();
		$this->load_elements();
	}

	/**
	 * Adding the proper hooks for ACSS version
	 *
	 * @return void
	 */
	public function add_hooks(): void {}

	/**
	 * Load Frames elements in the Bricks based on the ACSS settings
	 *
	 * @return void
	 */
	public function load_elements(): void {
		$acss_database = ACSS::get_settings();

		if ( ! $acss_database ) {
			Logger::log( sprintf( '%s: ACSS is not active to load the Frames elements', __METHOD__ ) );
			return;
		}

		foreach ( $this->widgets as $widget_name => $widget_options ) {
			$option = $widget_options['option'];
			$setting = $acss_database->get_var( $option );

			if ( ( null !== $setting && 'on' === $setting ) || ( array_key_exists( $widget_name, $this->default_widgets_states ) && 'off' !== $setting ) ) {
				if ( 'table-of-contents-v2' === $widget_name && ! defined( 'FRAMES_FLAG_ENABLE_NEW_TOC' ) ) {
					// Special case: set the 'FRAMES_FLAG_ENABLE_NEW_TOC' feature flag to true.
					Logger::log( sprintf( '%s: Enabling new Table of Contents', __METHOD__ ) );
					define( 'FRAMES_FLAG_ENABLE_NEW_TOC', true );
					continue;
				}
				if ( 'accordion-v2' === $widget_name && ! defined( 'FRAMES_FLAG_ENABLE_NEW_ACCORDION' ) ) { // TODO: uncomment to make it available on ACSS dashboard.
					Logger::log( sprintf( '%s: Enabling new Accordion', __METHOD__ ) );
					define( 'FRAMES_FLAG_ENABLE_NEW_ACCORDION', true );
					continue;
				} //.
				$element_file = FRAMES_WIDGETS_DIR . "/{$widget_name}/{$widget_name}.php";
				if ( file_exists( $element_file ) ) {
					Logger::log( sprintf( '%s: Registering element %s', __METHOD__, $widget_name ) );
					\Bricks\Elements::register_element( $element_file );
				} else {
					Logger::log( sprintf( '%s: Element file %s does not exist', __METHOD__, $element_file ) );
				}
			}
		}
	}
}
