<?php
/**
 * Frames Widget Manager class file.
 *
 * @package Frames_Client
 */

namespace Frames_Client;

use Frames_Client\Helpers\ACSS;
use Frames_Client\Helpers\Logger;
use Frames_Client\Traits\Singleton;
use Frames_Client\ACSS_Connectors\ACSS_Connector_Factory;

/**
 * Widget Manager class.
 */
final class Widget_Manager {

	use Singleton;

	/**
	 * Array of widgets.
	 *
	 * @var array
	 */
	private $widgets = array(
		/**
		 * 'widget-name' => array(
		 *   'option' => 'option-name',
		 *   'label' => 'Widget Label'
		 * )
		 */
		'table-of-contents' => array(
			'option' => 'option-frames-toc-widget',
			'label' => 'Table of Contents',
		),
		'table-of-contents-v2' => array(
			'option' => 'option-frames-toc-widget-v2',
			'label' => 'Use new Table of Contents ( Beta )',
		),
		'modal'             => array(
			'option' => 'option-frames-modal-widget',
			'label' => 'Modal',
		),
		'trigger'           => array(
			'option' => 'option-frames-trigger-widget',
			'label' => 'Trigger',
		),
		'slider'            => array(
			'option' => 'option-frames-slider-widget',
			'label' => 'Slider',
		),
		'slider-controls'   => array(
			'option' => 'option-frames-slider-controls-widget',
			'label' => 'Slider Controls',
		),
		'accordion'         => array(
			'option' => 'option-frames-accordion-widget',
			'label' => 'Accordion',
		),
		'accordion-v2'  => array( // TODO: uncomment to make it available on ACSS dashboard.
			'option' => 'option-frames-accordion-widget-v2',
			'label' => 'Use new Accordion ( Alpha )',
		),
		'tabs'              => array(
			'option' => 'option-frames-tabs-widget',
			'label' => 'Tabs',
		),
		'switch'            => array(
			'option' => 'option-frames-switch-widget',
			'label' => 'Switch',
		),
		'color-scheme'        => array(
			'option' => 'option-frames-color-scheme-widget',
			'label' => 'Color Scheme',
		),
		'notes'             => array(
			'option' => 'option-frames-notes-widget',
			'label' => 'Notes',
		),
	);

	/**
	 * Default states for widgets.
	 *
	 * @var array
	 */
	private $default_widgets_states = array(
		'notes' => 'on',
	);

	/**
	 * Initialize the Widgets.
	 *
	 * @return Widget_Manager
	 */
	public function init() {
		add_action( 'init', array( $this, 'init_widgets' ), 11 );

		return $this;
	}

	/**
	 * Initialize the Widgets.
	 *
	 * @return void
	 */
	public function init_widgets() {
		Logger::log( sprintf( '%s: Initializing widgets', __METHOD__ ) );
		if ( empty( $this->widgets ) || ! is_array( $this->widgets ) || ! class_exists( '\Bricks\Elements' ) ) {
			Logger::log( sprintf( '%s: No widgets to initialize', __METHOD__ ) );
			return;
		}

		if ( ! ACSS::is_ready() ) {
			Logger::log( sprintf( '%s: Automatic CSS is not ready', __METHOD__ ) );
			return;
		}

		$acss_version = ACSS::get_version() ? ACSS::get_version() : '';
		$acss_connector = ACSS_Connector_Factory::get_connector( $acss_version, $this->widgets, $this->default_widgets_states );
		$acss_connector->init();
	}

	/**
	 * Helper functions to control the output of the widget.
	 *
	 * @param array  $settings The settings array.
	 * @param string $control The control name.
	 * @param string $outputTrue The output if the control is true.
	 * @param string $outputFalse The output if the control is false.
	 * @since 1.2.0
	 */
	public static function ifControlIsTrue( $settings, $control, $outputTrue, $outputFalse ) { // phpcs:ignore
		if ( isset( $settings[ str_replace( ' ', '', $control ) ] ) && true === $settings[ str_replace( ' ', '', $control ) ] ) {
			echo $outputTrue; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		} else {
			echo $outputFalse; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}

	public static function ifControlIsEmpty( $settings, $control, $outputTrue, $outputFalse ) { // phpcs:ignore
		if ( isset( $settings[ str_replace( ' ', '', $control ) ] ) && '' === $settings[ str_replace( ' ', '', $control ) ] ) {
			echo $outputTrue; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		} else {
			echo $outputFalse; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}

	/**
	 * Undocumented function
	 *
	 * @param array  $settings The settings array.
	 * @param string $control The control name.
	 * @return string
	 * @since 1.2.0
	 */
	public static function control( $settings, $control ) {
		return isset( $settings[ str_replace( ' ', '', $control ) ] ) ? wp_kses_post( $settings[ str_replace( ' ', '', $control ) ] ) : '';
	}

	/**
	 * Takes a string of HTML attributes and merges them with an array of attributes.
	 *
	 * @param string $render_attributes The root attributes.
	 * @param array  $new_attributes The new attributes.
	 * @return string
	 */
	public static function render_attributes( $render_attributes, $new_attributes = array() ) {
		// Logger::log( sprintf( '%s: attributes=%s', __METHOD__, print_r( $render_attributes, true ) ) ); // TODO: remove.
		$attributes = array();
		// STEP: parse the attributes in an array.
		preg_match_all( '/([\w-]+)\s*=\s*"([^"]*)"/', $render_attributes, $matches, PREG_SET_ORDER );
		foreach ( $matches as $match ) {
			$attributes[ $match[1] ] = explode( ' ', $match[2] );
		}
		// STEP: merge the new attributes.
		$attributes = array_merge_recursive( $attributes, $new_attributes );
		// Logger::log( sprintf( '%s: attributes=%s', __METHOD__, print_r( $attributes, true ) ) ); // TODO: remove.
		// STEP: build the new attributes string.
		$render_attributes = '';
		foreach ( $attributes as $attribute_key => $attribute_value ) {
			if ( is_array( $attribute_value ) ) {
				$attribute_value = implode( ' ', array_unique( $attribute_value ) );
			}
			$render_attributes .= sprintf( ' %s="%s"', esc_attr( $attribute_key ), esc_attr( $attribute_value ) );
		}
		// Logger::log( sprintf( '%s: render_attributes=%s', __METHOD__, $render_attributes ) ); // TODO: remove.
		return $render_attributes;
	}

	/**
	 * Check if we're loading the frontend or the builder iframe.
	 *
	 * @return boolean
	 */
	public static function is_bricks_frontend() {
		if ( ! function_exists( 'bricks_is_frontend' ) || ! function_exists( 'bricks_is_builder_iframe' ) ) {
			Logger::log( sprintf( '%s: Bricks functions not found', __METHOD__ ) );
			return false;
		}
		return bricks_is_frontend() || bricks_is_builder_iframe();
	}

	/**
	 * Check if we're loading the builder.
	 * Condition to be used to load scripts only in the builder.
	 * Using builder_is_iframe will block from accessing __VUE_APP__
	 *
	 * @return boolean
	 */
	public static function is_bricks_builder() {
		if ( ! function_exists( 'bricks_is_builder' ) ) {
			Logger::log( sprintf( '%s: bricks_is_builder function not found', __METHOD__ ) );
			return false;
		}
		return bricks_is_builder();
	}
}
