<?php
/**
 * Render elements
 *
 * @package   WP Grid Builder - Bricks
 * @author    Loïc Blascos
 * @copyright 2019-2024 Loïc Blascos
 */

namespace WP_Grid_Builder_Bricks\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Render
 *
 * @class WP_Grid_Builder_Bricks\Includes\Render
 * @since 1.0.0
 */
final class Render {

	use Helpers;

	/**
	 * Holds Bricks elements to handle
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var array
	 */
	protected $elements = [];

	/**
	 * Holds Bricks rendered post IDs
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var array
	 */
	protected $post_ids = [];

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {

		add_filter( 'wp_grid_builder_bricks/handle_element', [ $this, 'handle_element' ], 10, 3 );
		add_filter( 'wp_grid_builder_bricks/setup_element', [ $this, 'setup_element' ] );
		add_filter( 'wp_grid_builder/frontend/register_scripts', [ $this, 'register_script' ] );
		add_filter( 'wp_grid_builder/frontend/add_inline_script', [ $this, 'inline_script' ] );
		add_filter( 'wp_grid_builder/frontend/localize_script', [ $this, 'unset_elements' ] );
		add_filter( 'bricks/builder/data_post_id', [ $this, 'get_elements' ] );
		add_filter( 'bricks/query/force_run', '__return_true', PHP_INT_MAX );
		add_action( 'wp', [ $this, 'get_active_templates' ], PHP_INT_MAX - 10 );

	}

	/**
	 * Whether to handle element
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param boolean $handle  Whether to handle element.
	 * @param string  $element Element id.
	 * @param boolean $ignore  Whether to unset/ignre element.
	 * @return boolean
	 */
	public function handle_element( $handle, $element, $ignore = false ) {

		if ( ! isset( $this->elements[ $element ] ) ) {
			return false;
		}

		// Prevent rendering duplicate elements.
		if ( ! $ignore ) {
			unset( $this->elements[ $element ] );
		}

		return true;

	}

	/**
	 * Setup element ID to handle
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param string $element_id Element ID to handle.
	 */
	public function setup_element( $element_id ) {

		if ( empty( $element_id ) || isset( $this->elements[ $element_id ] ) ) {
			return;
		}

		$this->elements[ $element_id ] = true;

	}

	/**
	 * Loop through elements to render
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param integer $post_id Holds current post ID.
	 * @return integer
	 */
	public function get_elements( $post_id ) {

		if ( bricks_is_builder() || wpgb_doing_ajax() ) {
			return $post_id;
		}

		if ( ! $post_id || in_array( $post_id, $this->post_ids, true ) ) {
			return $post_id;
		}

		array_push( $this->post_ids, $post_id );

		$elements = $this->get_bricks_data( $post_id );

		if ( false === $elements ) {
			return $post_id;
		}

		foreach ( $elements as $element ) {

			if ( empty( $element['id'] ) || empty( $element['name'] ) ) {
				continue;
			}

			$this->get_template( $element );

			if ( 'wpgb-facet' !== $element['name'] ) {
				continue;
			}

			if ( ! empty( $element['settings']['grid'] ) ) {
				$this->setup_element( $element['settings']['grid'] );
			} elseif ( ! empty( $element['settings']['element_id'] ) ) {
				$this->setup_element( 'bricks-element-' . $element['settings']['element_id'] );
			}
		}

		return $post_id;

	}

	/**
	 * Retrieve template elements to render
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $element Holds element.
	 */
	public function get_template( $element ) {

		if ( 'template' !== $element['name'] || empty( $element['settings']['template'] ) ) {
			return;
		}

		apply_filters( 'wp_grid_builder_bricks/data_post_id', $element['settings']['template'] );
		$this->get_elements( $element['settings']['template'] );

	}

	/**
	 * Retrieve active template elements to render
	 * Fixed an issue with Bricks v1.7.2, which returns the post ID to 0 for archive templates.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function get_active_templates() {

		if ( ! class_exists( '\Bricks\Database' ) ) {
			return;
		}

		if ( ! property_exists( '\Bricks\Database', 'active_templates' ) ) {
			return;
		}

		$templates = \Bricks\database::$active_templates;

		if ( ! is_array( $templates ) ) {
			return;
		}

		array_map(
			function( $template ) {
				if ( is_int( $template ) ) {
					apply_filters( 'wp_grid_builder_bricks/data_post_id', $this->get_elements( $template ) );
				}
			},
			$templates
		);
	}

	/**
	 * Unset shadow elements generated by Bricks facets
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $data Holds data to localize.
	 * @return array
	 */
	public function unset_elements( $data ) {

		if ( empty( $data['shadowGrids'] ) ) {
			return $data;
		}

		$data['shadowGrids'] = array_values(
			array_filter(
				$data['shadowGrids'],
				function( $id ) {
					return strpos( $id, 'bricks-element-' ) === false;
				}
			)
		);

		return $data;

	}

	/**
	 * Register script
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $scripts Holds script to register.
	 * @return array
	 */
	public function register_script( $scripts ) {

		$scripts[] = [
			'handle'  => 'wpgb-bricks',
			'source'  => WPGB_BRICKS_URL . 'assets/js/frontend.js',
			'version' => WPGB_BRICKS_VERSION,
		];

		return $scripts;

	}

	/**
	 * Add inline javascript code to instantiate Bricks elements
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param string $script Inline script.
	 * @return string
	 */
	public function inline_script( $script ) {

		$rendered = [];

		foreach ( Container::all() as $id => $instance ) {
			$rendered[] = $instance->inline_script( $script );
		}

		if ( ! empty( $rendered ) ) {
			$script .= implode( '', array_unique( $rendered ) );
		}

		return apply_filters( 'wp_grid_builder/template/inline_script', $script );

	}
}
