<?php
/**
 * Grid element
 *
 * @package   WP Grid Builder - Bricks
 * @author    Loïc Blascos
 * @copyright 2019-2024 Loïc Blascos
 */

namespace WP_Grid_Builder_Bricks\Elements;

use WP_Grid_Builder_Bricks\Includes\Helpers;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Grid element
 *
 * @class WP_Grid_Builder_Bricks\Elements\Grid
 * @since 1.0.0
 */
final class Grid extends \Bricks\Element {

	use Helpers;

	/**
	 * Category slug
	 *
	 * @since 1.0.0
	 * @access public
	 * @var string
	 */
	public $category = 'WP Grid Builder';

	/**
	 * Element name
	 *
	 * @since 1.0.0
	 * @access public
	 * @var string
	 */
	public $name = 'wpgb-grid';

	/**
	 * Element icon
	 *
	 * @since 1.0.0
	 * @access public
	 * @var string
	 */
	public $icon = 'ti-layout-grid2';

	/**
	 * Element CSS selector
	 *
	 * @since 1.0.0
	 * @access public
	 * @var string
	 */
	public $css_selector = '.wp-grid-builder';

	/**
	 * Nestable element
	 *
	 * @since 1.0.0
	 * @access public
	 * @var boolean
	 */
	public $nestable = false;

	/**
	 * Element label
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string
	 */
	public function get_label() {

		return esc_html__( 'Grid', 'wpgb-bricks' );

	}

	/**
	 * Element controls
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function set_controls() {

		include WPGB_BRICKS_PATH . 'includes/elements/controls/grid.php';

	}

	/**
	 * Widget render
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function render() {

		$is_empty = empty( $this->settings['id'] );

		if ( ! $this->is_builder() && $is_empty ) {
			return;
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo '<div ' . $this->render_attributes( '_root' ) . '>';
			! $is_empty && wpgb_render_grid(
				array_merge(
					[ 'id' => $this->settings['id'] ],
					$this->default_query()
				)
			);
			$this->enqueue_grid_assets();
		echo '</div>';

	}

	/**
	 * Get default editor query to simulate main query.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array
	 */
	public function default_query() {

		$query_args['is_main_query'] = ! empty( $this->settings['is_main_query'] );

		if ( ! $this->is_builder() ) {
			return $query_args;
		}

		if ( ! $query_args['is_main_query'] ) {
			return $query_args;
		}

		$post_types = get_post_types( [ 'public' => true ] );
		unset( $post_types['attachment'] );

		return [
			'post_type'   => $post_types,
			'post_status' => 'publish',
		];
	}

	/**
	 * Enqueue widget styles in editor
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function enqueue_grid_assets() {

		if ( ! $this->is_builder() ) {
			return;
		}

		// Enqueue plugin styles.
		wpgb_enqueue_styles();

		// Add inline styles generated by a grid only.
		$styles = wp_styles();
		$styles->print_inline_style( WPGB_SLUG . '-style' );
		$styles->do_item( WPGB_SLUG . '-grids' );
		$styles->do_item( WPGB_SLUG . '-fonts' );

	}
}
