<?php
/**
 * Async
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handle asynchonous request
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Async
 * @since 1.0.0
 */
abstract class Async {

	/**
	 * Marker settings
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var array
	 */
	protected $marker;

	/**
	 * Constructor
	 *
	 * @since 2.0.0
	 * @access public
	 */
	public function __construct() {

		add_action( 'rest_api_init', [ $this, 'register_route' ] );

	}

	/**
	 * Register REST route
	 *
	 * @since 2.0.0
	 * @access public
	 */
	public function register_route() {

		register_rest_route(
			'wpgb_map/v2',
			'popup',
			[
				'methods'             => \WP_REST_Server::READABLE,
				'callback'            => [ $this, 'maybe_handle' ],
				'permission_callback' => '__return_true',
				'args'                => [
					'id'     => [
						'type'              => 'integer',
						'required'          => true,
						'sanitize_callback' => 'absint',
					],
					'lat'    => [
						'type'     => 'number',
						'required' => true,
					],
					'lng'    => [
						'type'     => 'number',
						'required' => true,
					],
					'lang'   => [
						'type'              => 'string',
						'default'           => '',
						'sanitize_callback' => 'sanitize_text_field',
					],
					'facet'  => [
						'type'              => 'integer',
						'required'          => true,
						'sanitize_callback' => 'absint',
					],
					'source' => [
						'type'     => 'string',
						'required' => true,
						'enum'     => [ 'post_type', 'term', 'user' ],
					],
				],

			]
		);
	}

	/**
	 * Send response
	 *
	 * @since 2.0.0
	 * @access protected
	 *
	 * @param boolean $success    Success state.
	 * @param string  $content    Holds content to render.
	 * @param string  $attributes Holds marker attributes.
	 * @return WP_REST_Response
	 */
	protected function send_response( $success = true, $content = '', $attributes = [] ) {

		return rest_ensure_response(
			[
				'success'    => (bool) $success,
				'content'    => $content,
				'attributes' => $attributes,
			]
		);
	}

	/**
	 * Handle request.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response
	 */
	public function maybe_handle( $request ) {

		$this->marker = $request->get_params();
		$content      = $this->query();

		if ( is_wp_error( $content ) ) {

			return $this->send_response(
				false,
				__( 'Sorry, an unknown error occurred.', 'wpgb-map-facet' )
			);
		}

		return $this->send_response(
			true,
			$content,
			$this->get_card_attributes()
		);
	}

	/**
	 * Handle query request
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	abstract protected function query();
}
