<?php
/**
 * Card
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Marker card
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Card
 * @since 2.0.0
 */
trait Card {

	/**
	 * Card popup attributes
	 *
	 * @since 2.0.0
	 * @access public
	 * @var array
	 */
	public $attributes = [];

	/**
	 * Get facet cards
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param integer $facet Facet ID.
	 * @return array
	 */
	final public function get_cards( $facet ) {

		global $wpdb;

		$facet = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT settings FROM {$wpdb->prefix}wpgb_facets WHERE id = %d",
				$facet
			)
		);

		if ( empty( $facet ) ) {
			return [];
		}

		$settings = json_decode( $facet, true );

		if ( empty( $settings['map_popups'] ) ) {
			return [];
		}

		$cards = array_filter(
			$settings['map_popups'],
			function( $condition ) {
				return ! empty( $condition['card'] );
			}
		);

		if ( empty( $cards ) ) {
			return [];
		}

		return $cards;

	}

	/**
	 * Get card attributes
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array
	 */
	final public function get_card_attributes() {

		return $this->attributes;

	}

	/**
	 * Set card attributes
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param integer $card Card ID.
	 * @return integer
	 */
	final public function set_card_attributes( $card = 0 ) {

		$card_types = wpgb_get_grid_settings( 'card_types' );

		foreach ( (array) $card_types as $type ) {

			if ( empty( $type['card'] ) ) {
				continue;
			}

			if (
				// If no conditions, assign card to any content type.
				empty( $type['conditions'] ) ||
				apply_filters( 'wp_grid_builder/do_conditions', true, $type['conditions'], 'card' )
			) {

				if ( ! empty( $type['attributes'] ) ) {
					$this->attributes = $type['attributes'];
				}

				if ( empty( $this->attributes['width'] ) ) {
					$this->attributes = [ 'width' => 300 ];
				}

				break;

			}
		}

		return $card;

	}

	/**
	 * Render card
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param array $marker Holds marker arguments.
	 * @return string
	 */
	final public function render_card( $marker ) {

		$cards = $this->get_cards( $marker['facet'] );

		if ( empty( $cards ) ) {
			return '';
		}

		ob_start();
		add_filter( 'wp_grid_builder/card/id', [ $this, 'set_card_attributes' ] );
		wpgb_render_grid(
			[
				'id'                  => 0,
				'is_marker'           => true,
				'is_dynamic'          => true,
				'source'              => $marker['source'],
				'lang'                => $marker['lang'],
				'posts_per_page'      => 1,
				'post__in'            => [ $marker['id'] ],
				'term__in'            => [ $marker['id'] ],
				'user__in'            => [ $marker['id'] ],
				'post_status'         => 'any',
				'post_type'           => get_post_types(),
				'card_types'          => $cards,
				'hide_empty'          => false,
				'ignore_sticky_posts' => true,
			],
			'Loop'
		);

		$card = ob_get_clean();

		if ( strpos( $card, 'wpgb-card-default' ) !== false ) {
			return '';
		}

		return $card;

	}

	/**
	 * Enqueue cards
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param integer $facet Facet ID.
	 */
	final public function enqueue_cards( $facet ) {

		$cards = $this->get_cards( $facet );

		if ( empty( $cards ) ) {
			return;
		}

		wpgb_render_grid(
			[
				'id'         => '',
				'is_dynamic' => true,
				'card_types' => $cards,
				'card_sizes' => [],
			],
			'Assets'
		);
	}
}
