<?php
/**
 * Marker conditions
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handle conditions
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Conditions
 * @since 2.0.0
 */
final class Conditions {

	use Data;

	/**
	 * Constructor
	 *
	 * @since 2.0.0
	 * @access public
	 */
	public function __construct() {

		add_filter( 'wp_grid_builder/conditions', [ $this, 'conditions_context' ] );
		add_filter( 'wp_grid_builder/condition/value', [ $this, 'post_conditions' ], 10, 4 );
		add_filter( 'wp_grid_builder/condition/value', [ $this, 'term_conditions' ], 10, 4 );
		add_filter( 'wp_grid_builder/condition/value', [ $this, 'user_conditions' ], 10, 4 );

	}

	/**
	 * Add map context for conditions
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param array $conditions Holds conditions.
	 * @return array
	 */
	public function conditions_context( $conditions ) {

		$allowed = [ 'post', 'term', 'user', 'current_user', 'date', 'url', 'device' ];

		foreach ( $conditions as $group => &$condition ) {

			if ( in_array( $group, $allowed, true ) ) {

				$condition['options'] = array_map(
					function( $option ) {

						$option['context'][] = 'map';
						return $option;

					},
					$condition['options']
				);
			}
		}

		return $conditions;

	}

	/**
	 * Post conditions
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param string $return    Returned value.
	 * @param string $key       key condition.
	 * @param array  $condition Condition arguments.
	 * @param string $context   Condition context.
	 * @return mixed
	 */
	public function post_conditions( $return, $key, $condition, $context ) {

		global $wpgb_marker_id;

		if ( 'marker' !== $context ) {
			return $return;
		}

		$post = get_post( $wpgb_marker_id );

		if ( empty( $post ) ) {
			return $return;
		}

		switch ( $key ) {
			case 'post_id':
				return $post->ID ?? 0;
			case 'post_parent':
				return $post->post_parent ?? 0;
			case 'post_name':
				return $post->post_name ?? '';
			case 'post_title':
				return $post->post_title ?? '';
			case 'post_author':
				return $post->post_author ?? 0;
			case 'post_permalink':
				return get_the_permalink( $post );
			case 'post_date':
				return isset( $post->post_date ) ? date_i18n( 'Y-m-d', strtotime( $post->post_date ) ) : '';
			case 'post_modified_date':
				return isset( $post->post_modified ) ? date_i18n( 'Y-m-d', strtotime( $post->post_modified ) ) : '';
			case 'post_content':
				return $post->post_content ?? '';
			case 'post_excerpt':
				return $post->post_excerpt ?? '';
			case 'post_status':
				return $post->post_status ?? '';
			case 'post_type':
				return $post->post_type ?? '';
			case 'post_format':
				return get_post_format( $post );
			case 'post_thumbnail':
				return has_post_thumbnail( $post );
			case 'post_comments_number':
				return $post->comment_count ?? 0;
			case 'post_metadata':
				return $this->metadata( 'post', $wpgb_marker_id, trim( $condition['field'] ?? '' ) );
			case 'post_term':
				return (array) (
					new \WP_Term_Query(
						[
							'object_ids' => $wpgb_marker_id,
							'include'    => $condition['value'],
							'fields'     => 'ids',
						]
					)
				)->get_terms();
		}

		return $return;

	}

	/**
	 * Term conditions
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param string $return    Returned value.
	 * @param string $key       key condition.
	 * @param array  $condition Condition arguments.
	 * @param string $context   Condition context.
	 * @return mixed
	 */
	public function term_conditions( $return, $key, $condition, $context ) {

		global $wpgb_marker_id;

		if ( 'marker' !== $context ) {
			return $return;
		}

		$term = get_term( $wpgb_marker_id );

		if ( empty( $term ) ) {
			return $return;
		}

		switch ( $key ) {
			case 'term_id':
				return $term->term_id ?? 0;
			case 'term_slug':
				return $term->slug ?? '';
			case 'term_name':
				return $term->name ?? '';
			case 'term_taxonomy':
				return $term->taxonomy ?? '';
			case 'term_parent':
				return $term->parent ?? 0;
			case 'term_description':
				return $term->description ?? '';
			case 'term_count':
				return $term->count ?? 0;
			case 'term_metadata':
				return $this->metadata( 'term', $wpgb_marker_id, trim( $condition['field'] ?? '' ) );
		}

		return $return;

	}

	/**
	 * User conditions
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param string $return    Returned value.
	 * @param string $key       Key condition.
	 * @param array  $condition Condition arguments.
	 * @param string $context   Condition context.
	 * @return mixed
	 */
	public function user_conditions( $return, $key, $condition, $context ) {

		global $wpgb_marker_id;

		if ( 'marker' !== $context ) {
			return $return;
		}

		$user = get_user_by( 'ID', $wpgb_marker_id );

		if ( empty( $user ) ) {
			return $return;
		}

		switch ( $key ) {
			case 'user_id':
				return $user->ID ?? 0;
			case 'user_login':
				return $user->user_login ?? '';
			case 'user_display_name':
				return $user->display_name ?? '';
			case 'user_first_name':
				return $user->first_name ?? '';
			case 'user_last_name':
				return $user->last_name ?? '';
			case 'user_nickname':
				return $user->nickname ?? '';
			case 'user_description':
				return $user->description ?? '';
			case 'user_email':
				return $user->user_email ?? '';
			case 'user_url':
				return $user->user_url ?? '';
			case 'user_roles':
				return $user->roles ?? [];
			case 'user_post_count':
				return count_user_posts( $user->ID ) ?? 0;
			case 'user_metadata':
				return $this->metadata( 'user', $wpgb_marker_id, trim( $condition['field'] ?? '' ) );
		}

		return $this->user_data( $return, $key, $wpgb_marker_id, $condition );

	}

	/**
	 * Meta conditions
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param string  $meta_type Metadata type.
	 * @param integer $object_id Object ID.
	 * @param string  $meta_key  MetaData key.
	 * @return mixed
	 */
	public function metadata( $meta_type, $object_id, $meta_key ) {

		if ( empty( $meta_key ) ) {
			return '';
		}

		$value = apply_filters( 'wp_grid_builder/metadata', '', $meta_type, $object_id, $meta_key );

		if ( ! is_null( $value ) && '' !== $value && false !== $value ) {
			return $value;
		}

		return wpgb_format_metadata( [], get_metadata( $meta_type, $object_id, $meta_key, true ) );

	}
}
