<?php
/**
 * Marker data
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handle marker data
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Data
 * @since 2.0.0
 */
trait Data {

	/**
	 * Get post data
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param string  $return    Returned value.
	 * @param string  $data      Data to retrieve.
	 * @param integer $object_id Object ID.
	 * @param array   $args      Data arguments.
	 * @return mixed
	 */
	final public function post_data( $return, $data, $object_id, $args ) {

		global $wp_post_statuses, $wp_post_types;

		if ( empty( $object_id ) ) {
			return $return;
		}

		$post = get_post( $object_id );

		if ( empty( $post ) ) {
			return $return;
		}

		switch ( $data ) {
			case 'post_id':
				return $post->ID ?? 0;
			case 'post_name':
				return $post->post_name ?? '';
			case 'post_title':
				return $post->post_title ?? '';
			case 'post_author':
				return get_the_author_meta( 'display_name', $post->post_author ) ?? '';
			case 'post_permalink':
				return get_the_permalink( $post );
			case 'post_date':
				return get_the_date( '', $post ) ?? '';
			case 'post_modified_date':
				return get_the_modified_date( '', $post ) ?? '';
			case 'post_content':
				return $post->post_content ?? '';
			case 'post_excerpt':
				return $post->post_excerpt ?? '';
			case 'post_status':
				return $wp_post_statuses[ $post->post_status ]->label ?? $post->post_status ?? '';
			case 'post_type':
				return $wp_post_types[ $post->post_type ]->labels->singular_name ?? $post->post_type ?? '';
			case 'post_format':
				return get_post_format_string( get_post_format( $post ) );
			case 'post_comments_number':
				return $post->comment_count ?? 0;
			case 'post_metadata':
				return $this->metadata( 'post', $object_id, trim( $args['field'] ?? '' ) );
		}

		return $return;

	}

	/**
	 * Get term data
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param string  $return    Returned value.
	 * @param string  $data      Data to retrieve.
	 * @param integer $object_id Object ID.
	 * @param array   $args      Data arguments.
	 * @return mixed
	 */
	final public function term_data( $return, $data, $object_id, $args ) {

		if ( empty( $object_id ) ) {
			return $return;
		}

		$term = get_term( $object_id );

		if ( empty( $term ) ) {
			return $return;
		}

		switch ( $data ) {
			case 'term_id':
				return $term->term_id ?? 0;
			case 'term_slug':
				return $term->slug ?? '';
			case 'term_name':
				return $term->name ?? '';
			case 'term_taxonomy':
				return $term->taxonomy ?? '';
			case 'term_parent':
				return $term->parent ?? 0;
			case 'term_description':
				return $term->description ?? '';
			case 'term_count':
				return $term->count ?? 0;
			case 'term_metadata':
				return $this->metadata( 'term', $object_id, trim( $args['field'] ?? '' ) );
		}

		return $return;

	}

	/**
	 * Get user data
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param string  $return    Returned value.
	 * @param string  $data      Data to retrieve.
	 * @param integer $object_id Object ID.
	 * @param array   $args      Data arguments.
	 * @return mixed
	 */
	final public function user_data( $return, $data, $object_id, $args ) {

		if ( empty( $object_id ) ) {
			return $return;
		}

		$user = get_user_by( 'ID', $object_id );

		if ( empty( $user ) ) {
			return $return;
		}

		switch ( $data ) {
			case 'user_id':
				return $user->ID ?? 0;
			case 'user_login':
				return $user->user_login ?? '';
			case 'user_display_name':
				return $user->display_name ?? '';
			case 'user_first_name':
				return $user->first_name ?? '';
			case 'user_last_name':
				return $user->last_name ?? '';
			case 'user_nickname':
				return $user->nickname ?? '';
			case 'user_description':
				return $user->description ?? '';
			case 'user_email':
				return $user->user_email ?? '';
			case 'user_url':
				return $user->user_url ?? '';
			case 'user_roles':
				return $this->get_user_roles_translated( $user->roles ?? [] );
			case 'user_post_count':
				return count_user_posts( $user->ID ) ?? 0;
			case 'user_metadata':
				return $this->metadata( 'user', $object_id, trim( $args['field'] ?? '' ) );
		}

		return $return;

	}

	/**
	 * Meta data
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param string  $meta_type Metadata type.
	 * @param integer $object_id Object ID.
	 * @param string  $meta_key  MetaData key.
	 * @return mixed
	 */
	final public function metadata( $meta_type, $object_id, $meta_key ) {

		if ( empty( $meta_key ) ) {
			return '';
		}

		$value = apply_filters( 'wp_grid_builder/metadata', '', $meta_type, $object_id, $meta_key );

		if ( ! is_null( $value ) && '' !== $value && false !== $value ) {
			return $value;
		}

		return wpgb_format_metadata( [], get_metadata( $meta_type, $object_id, $meta_key, true ) );

	}

	/**
	 * Get translated user roles
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param array $roles User roles.
	 * @return string
	 */
	final public function get_user_roles_translated( $roles ) {

		static $domain_loaded = false;

		if ( empty( $roles ) ) {
			return '';
		}

		// Role translations are not loaded on frontend.
		if ( ! $domain_loaded && ! is_admin() ) {

			load_textdomain( 'default', WP_LANG_DIR . '/admin-' . get_locale() . '.mo' );
			$domain_loaded = true;

		}

		if ( ! function_exists( 'get_editable_roles' ) ) {
			require_once ABSPATH . 'wp-admin/includes/user.php';
		}

		$editable_roles = get_editable_roles();

		return join(
			', ',
			array_map(
				function( $role ) use ( $editable_roles ) {

					if ( ! empty( $editable_roles[ $role ]['name'] ) ) {
						return translate_user_role( $editable_roles[ $role ]['name'] );
					}

					return $role;

				},
				(array) $roles
			)
		);
	}
}
