<?php
/**
 * Facets
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main Instance of the plugin
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Facets
 * @since 1.0.0
 */
final class Facets extends Async {

	use Query;

	/**
	 * Holds facet identifiers
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var array
	 */
	protected $facets = [];

	/**
	 * Holds grid identifiers
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var array
	 */
	protected $grids = [];

	/**
	 * Holds facet types
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var array
	 */
	protected $types;

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {

		parent::__construct();

		add_filter( 'wp_grid_builder/facets', [ $this, 'register_facets' ], 99 );
		add_filter( 'wp_grid_builder/controls/style', [ $this, 'register_styles' ], 99 );
		add_filter( 'wp_grid_builder/custom_fields', [ $this, 'custom_field' ] );
		add_filter( 'wp_grid_builder/indexer/row', [ $this, 'index_map' ], 10, 3 );
		add_filter( 'wp_grid_builder/facet/render_args', [ $this, 'render_facet' ] );
		add_action( 'wp_footer', [ $this, 'register_cards' ], 0 );
		add_filter( 'wp_grid_builder/frontend/register_styles', [ $this, 'register_cards' ] );
		add_filter( 'wp_grid_builder/frontend/register_styles', [ $this, 'register_style' ], 0 );
		add_filter( 'wp_grid_builder/frontend/register_scripts', [ $this, 'register_scripts' ] );
		add_filter( 'wp_grid_builder/frontend/enqueue_styles', [ $this, 'enqueue_assets' ] );
		add_filter( 'wp_grid_builder/frontend/enqueue_scripts', [ $this, 'enqueue_assets' ] );
		add_filter( 'wp_grid_builder/frontend/localize_script', [ $this, 'localize_script' ] );

	}

	/**
	 * Register facets (proximity facet will come later)
	 *
	 * @since 1.1.0 Added geolocation facet
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $facets Holds registered facet arguments.
	 * @return array
	 */
	public function register_facets( $facets ) {

		$facets['geolocation'] = [
			'name'     => __( 'Geolocation', 'wpgb-map-facet' ),
			'type'     => 'filter',
			'class'    => __NAMESPACE__ . '\Facets\Geo',
			'icon'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="20" height="20" aria-hidden="true" focusable="false"><path d="M12 4c-4.4 0-8 3.6-8 8s3.6 8 8 8 8-3.6 8-8-3.6-8-8-8zm0 14.5c-3.6 0-6.5-2.9-6.5-6.5S8.4 5.5 12 5.5s6.5 2.9 6.5 6.5-2.9 6.5-6.5 6.5zM9 16l4.5-3L15 8.4l-4.5 3L9 16z"/></svg>',
			'icons'    => [
				'large' => WPGB_MAP_URL . 'assets/svg/sprite.svg?v=' . WPGB_MAP_VERSION . '#wpgb-geolocation-large-icon',
				'small' => WPGB_MAP_URL . 'assets/svg/sprite.svg?v=' . WPGB_MAP_VERSION . '#wpgb-geolocation-small-icon',
			],
			'controls' => include WPGB_MAP_PATH . '/includes/controls/geolocation.php',
		];

		$facets['map'] = [
			'name'     => __( 'Map', 'wpgb-map-facet' ),
			'type'     => 'filter',
			'class'    => __NAMESPACE__ . '\Facets\Map',
			'icon'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="20" height="20" aria-hidden="true" focusable="false"><path d="M12 9c-.8 0-1.5.7-1.5 1.5S11.2 12 12 12s1.5-.7 1.5-1.5S12.8 9 12 9zm0-5c-3.6 0-6.5 2.8-6.5 6.2 0 .8.3 1.8.9 3.1.5 1.1 1.2 2.3 2 3.6.7 1 3 3.8 3.2 3.9l.4.5.4-.5c.2-.2 2.6-2.9 3.2-3.9.8-1.2 1.5-2.5 2-3.6.6-1.3.9-2.3.9-3.1C18.5 6.8 15.6 4 12 4zm4.3 8.7c-.5 1-1.1 2.2-1.9 3.4-.5.7-1.7 2.2-2.4 3-.7-.8-1.9-2.3-2.4-3-.8-1.2-1.4-2.3-1.9-3.3-.6-1.4-.7-2.2-.7-2.5 0-2.6 2.2-4.7 5-4.7s5 2.1 5 4.7c0 .2-.1 1-.7 2.4z"/></svg>',
			'icons'    => [
				'large' => WPGB_MAP_URL . 'assets/svg/sprite.svg?v=' . WPGB_MAP_VERSION . '#wpgb-map-large-icon',
				'small' => WPGB_MAP_URL . 'assets/svg/sprite.svg?v=' . WPGB_MAP_VERSION . '#wpgb-map-small-icon',
			],
			'controls' => include WPGB_MAP_PATH . '/includes/controls/map.php',
		];

		return $facets;

	}

	/**
	 * Register card styles from Map facets (frontend and preview)
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param array $styles Holds registered facet style.
	 * @return array
	 */
	public function register_cards( $styles ) {

		if ( ! $this->has_facet( 'map' ) ) {
			return $styles;
		}

		if ( empty( $this->types['map'] ) ) {
			return $styles;
		}

		foreach ( $this->types['map'] as $facet ) {
			$this->enqueue_cards( $facet );
		}

		return $styles;

	}

	/**
	 * Register facet styles
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param array $styles Holds registered facet style.
	 * @return array
	 */
	public function register_styles( $styles ) {

		return array_merge(
			$styles,
			include WPGB_MAP_PATH . '/includes/controls/styles.php'
		);
	}

	/**
	 * Register custom field
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $fields Holds registered custom fields.
	 * @return array
	 */
	public function custom_field( $fields ) {

		$fields['Map Facet'] = [
			'_wpgb_map_coordinates' => __( 'Map Coordinates', 'wpgb-map-facet' ),
		];

		return $fields;

	}

	/**
	 * Get all facet identifiers.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $args Holds facet args.
	 * @return array
	 */
	public function render_facet( $args ) {

		$this->facets[] = $args['id'];
		$this->grids[]  = $args['grid'];

		return $args;

	}

	/**
	 * Check if the page contains map or geolocation facets
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param string $type Fact type to check.
	 * @return boolean
	 */
	public function has_facet( $type = '' ) {

		$this->get_types();

		$has_facet = wpgb_is_gutenberg();

		if ( ! empty( $this->types[ $type ] ) ) {
			$has_facet = true;
		}

		if ( in_array( 'wpgb_facet_preview', $this->grids, true ) ) {
			$has_facet = true;
		}

		return apply_filters( 'wp_grid_builder_map/has_facet', $has_facet, $type );

	}

	/**
	 * Get facet types
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function get_types() {

		global $wpdb;

		if ( isset( $this->types ) ) {
			return;
		}

		$this->types = [];

		if ( empty( $this->facets ) ) {
			return;
		}

		$this->facets = array_unique( $this->facets );
		$placeholders = rtrim( str_repeat( '%s,', count( $this->facets ) ), ',' );

		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id, type
				FROM {$wpdb->prefix}wpgb_facets
				WHERE type IN ('map','geolocation')
				AND id IN ($placeholders)", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$this->facets
			),
			'ARRAY_A'
		);

		foreach ( $results as $facet ) {
			$this->types[ $facet['type'] ][] = $facet['id'];
		}
	}

	/**
	 * Index map fields
	 *
	 * @since 1.0.4 Allow to index any array containing lat and lng properties.
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array   $row       Holds index row of the current object id.
	 * @param integer $object_id Post, term or user id to index.
	 * @param array   $facet     Holds facet settings.
	 * @return array
	 */
	public function index_map( $row, $object_id, $facet ) {

		if ( empty( $facet['filter_type'] ) || ! in_array( $facet['filter_type'], [ 'map', 'geolocation' ], true ) ) {
			return $row;
		}

		$has_coordinates = isset(
			$row['facet_value']['lat'],
			$row['facet_name']['lng']
		);

		if ( $has_coordinates ) {

			$row['facet_value'] = $row['facet_value']['lat'];
			$row['facet_name']  = $row['facet_name']['lng'];

		}

		return $row;

	}

	/**
	 * Register facet style
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $styles Holds styles to register.
	 * @return array
	 */
	public function register_style( $styles ) {

		$source = WPGB_MAP_URL . 'assets/css/style.css';

		if ( function_exists( 'wpgb_print_facet_style' ) ) {
			$source = WPGB_MAP_URL . 'assets/css/map.css';
		}

		$styles[] = [
			'handle'  => 'wpgb-map',
			'source'  => $source,
			'version' => WPGB_MAP_VERSION,
		];

		return $styles;

	}

	/**
	 * Register facet script
	 *
	 * @since 1.1.0 Added geolocation script
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $scripts Holds scripts to register.
	 * @return array
	 */
	public function register_scripts( $scripts ) {

		$scripts[] = [
			'handle'  => 'wpgb-geo-js',
			'source'  => WPGB_MAP_URL . 'assets/js/geo.js',
			'version' => WPGB_MAP_VERSION,
		];

		$scripts[] = [
			'handle'  => 'wpgb-map-js',
			'source'  => WPGB_MAP_URL . 'assets/js/map.js',
			'version' => WPGB_MAP_VERSION,
		];

		return $scripts;

	}

	/**
	 * Enqueue assets
	 *
	 * @since 1.1.0 Added check for geolocation facet type
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $handles Holds assets handle.
	 * @return array
	 */
	public function enqueue_assets( $handles ) {

		$has_map = $this->has_facet( 'map' );
		$has_geo = $this->has_facet( 'geolocation' );

		if ( ! $has_map ) {
			unset( $handles['wpgb-map-js'] );
		}

		if ( ! $has_geo ) {
			unset( $handles['wpgb-geo-js'] );
		}

		if ( ! $has_map && ! $has_geo ) {
			unset( $handles['wpgb-map'] );
		}

		return $handles;

	}

	/**
	 * Localize facet data
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $data Holds data to localize.
	 * @return array
	 */
	public function localize_script( $data ) {

		if ( $this->has_facet( 'map' ) ) {

			$data['mapFacet'] = [
				'restUrl'   => get_rest_url( null, 'wpgb_map/v2/popup/' ),
				'restNonce' => is_user_logged_in() ? wp_create_nonce( 'wp_rest' ) : false,
				'markers'   => esc_url( WPGB_MAP_URL . 'assets/imgs/m' ),
				'vendors'   => [
					[
						'type'    => 'js',
						'handle'  => 'wpgb-map',
						'source'  => esc_url( WPGB_MAP_URL . 'assets/js/vendors/leaflet.js' ),
						'version' => filemtime( WPGB_MAP_PATH . 'assets/js/vendors/leaflet.js' ),
					],
					[
						'type'    => 'css',
						'handle'  => 'wpgb-map',
						'source'  => esc_url( WPGB_MAP_URL . 'assets/css/vendors/leaflet.css' ),
						'version' => filemtime( WPGB_MAP_PATH . 'assets/css/vendors/leaflet.css' ),
					],
				],
			];
		}

		return $data;

	}
}
