<?php
/**
 * Query
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Query marker content
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Query
 * @since 1.0.0
 */
trait Query {

	use Card;

	/**
	 * Query marker content
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 *  @return string
	 */
	final public function query() {

		do_action( 'wp_grid_builder_map/before_marker_content', $this->marker );

		$content = $this->render_card( $this->marker );

		if ( empty( $content ) ) {

			switch ( $this->marker['source'] ) {
				case 'user':
					$content = $this->query_user();
					break;
				case 'term':
					$content = $this->query_term();
					break;
				default:
					$content = $this->query_post();
			}
		}

		if ( is_wp_error( $content ) ) {
			return $content;
		}

		do_action( 'wp_grid_builder_map/after_marker_content', $this->marker );

		$content = apply_filters( 'wp_grid_builder_map/marker_content', $content, $this->marker );

		wp_reset_postdata();

		return $content;

	}

	/**
	 * Query post
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 *  @return mixed
	 */
	final public function query_post() {

		$query = new \WP_Query(
			[
				'lang'                   => $this->marker['lang'],
				'post_status'            => 'any',
				'post_type'              => get_post_types(),
				'post__in'               => [ $this->marker['id'] ],
				'update_post_meta_cache' => false,
				'update_post_term_cache' => false,
				'no_found_rows'          => true,
				'ignore_sticky_posts'    => true,
			]
		);

		if ( ! $query->have_posts() ) {
			return new \WP_Error();
		}

		while ( $query->have_posts() ) {
			return include WPGB_MAP_PATH . 'includes/templates/post.php';
		}
	}

	/**
	 * Query user
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return mixed
	 */
	final public function query_user() {

		$query = new \WP_User_Query(
			[
				'number'      => 1,
				'include'     => [ $this->marker['id'] ],
				'count_total' => false,
			]
		);

		$users = $query->get_results();

		if ( empty( $users ) ) {
			return new \WP_Error();
		}

		foreach ( $users as $user ) {
			return include WPGB_MAP_PATH . 'includes/templates/user.php';
		}
	}

	/**
	 * Query user
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return mixed
	 */
	final public function query_term() {

		$query = new \WP_Term_Query(
			[
				'number'                 => 1,
				'lang'                   => $this->marker['lang'],
				'include'                => [ $this->marker['id'] ],
				'hide_empty'             => false,
				'update_term_meta_cache' => false,
			]
		);

		$terms = $query->terms;

		if ( empty( $terms ) ) {
			return new \WP_Error();
		}

		foreach ( $terms as $term ) {
			return include WPGB_MAP_PATH . 'includes/templates/term.php';
		}
	}
}
