<?php
/**
 * Facet Settings
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handle facet settings
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Settings
 * @since 1.0.0
 */
final class Settings {

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {

		add_filter( 'wp_grid_builder/settings/post_tabs', [ $this, 'add_post_tab' ] );
		add_filter( 'wp_grid_builder/tabs/post', [ $this, 'register_tab' ], 99, 1 );
		add_filter( 'wp_grid_builder/settings/post_fields', [ $this, 'add_post_fields' ] );
		add_filter( 'wp_grid_builder/settings/facet_fields', [ $this, 'add_facet_fields' ] );
		add_filter( 'wp_grid_builder/controls/post', [ $this, 'register_controls' ], 99, 1 );
		add_filter( 'wp_grid_builder/defaults/facet', [ $this, 'default_values' ], 99, 1 );
		add_filter( 'wp_grid_builder/settings/save_fields', [ $this, 'save_fields' ], 10, 3 );

	}

	/**
	 * Register post tab
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $tabs Holds post settings tabs.
	 * @return array
	 */
	public function add_post_tab( $tabs ) {

		$tabs[] = [
			'id'    => 'map_coordinates',
			'label' => __( 'Coordinates', 'wpgb-map-facet' ),
			'icon'  => WPGB_MAP_URL . 'assets/svg/sprite.svg?v=' . WPGB_MAP_VERSION . '#wpgb-pin-icon',
		];

		return $tabs;

	}

	/**
	 * Register tab controls
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @param array $tabs Holds registered tabs.
	 * @return array
	 */
	public function register_tab( $tabs = [] ) {

		array_push(
			$tabs,
			[
				'name'  => 'map_coordinates',
				'title' => __( 'Coordinates', 'wpgb-map-facet' ),
			]
		);

		return $tabs;

	}

	/**
	 * Register post fields
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $fields Holds post fields.
	 * @return array
	 */
	public function add_post_fields( $fields ) {

		$fields[] = include WPGB_MAP_PATH . 'includes/fields/post.php';

		return $fields;

	}

	/**
	 * Register facet fields
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $fields Holds facet fields.
	 * @return array
	 */
	public function add_facet_fields( $fields ) {

		return array_merge(
			$fields,
			include WPGB_MAP_PATH . 'includes/fields/api.php',
			include WPGB_MAP_PATH . 'includes/fields/appearance.php',
			include WPGB_MAP_PATH . 'includes/fields/layers.php',
			include WPGB_MAP_PATH . 'includes/fields/coordinates.php',
			include WPGB_MAP_PATH . 'includes/fields/behaviour.php',
			include WPGB_MAP_PATH . 'includes/fields/controls.php',
			include WPGB_MAP_PATH . 'includes/fields/marker.php'
		);
	}

	/**
	 * Register coordinates controls
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @param array $controls Holds registered controls.
	 * @return array
	 */
	public function register_controls( $controls = [] ) {

		return array_merge(
			$controls,
			include WPGB_MAP_PATH . '/includes/controls/post.php'
		);
	}

	/**
	 * Register default control values
	 *
	 * @since 1.5.0
	 * @access public
	 *
	 * @param array $values Holds default values.
	 * @return array
	 */
	public function default_values( $values = [] ) {

		return array_merge(
			$values,
			[
				'map_type'                  => 'leaflet',
				'map_access_token'          => '',
				'map_api_key'               => '',
				'map_limit_results'         => 'all',
				'map_google_style'          => 'roadmap',
				'map_mapbox_style'          => 'mapbox://styles/mapbox/streets-v11',
				'map_mapbox_style_url'      => '',
				'map_leaflet_style'         => 'Wikimedia',
				'map_ratio'                 => [
					'x' => 16,
					'y' => 9,
				],
				'map_lat'                   => 0,
				'map_lng'                   => 0,
				'map_zoom'                  => 2,
				'map_min_zoom'              => 0,
				'map_max_zoom'              => 20,
				'map_dragging'              => true,
				'map_scrolling'             => true,
				'map_filtering'             => false,
				'map_pan_search_ctrl'       => false,
				'map_pan_search_ctrl_label' => __( 'Search as I move the map', 'wpgb-map-facet' ),
				'map_scale_ctrl'            => true,
				'map_zoom_ctrl'             => true,
				'map_rotate_ctrl'           => true,
				'map_fullscreen_ctrl'       => true,
				'map_type_ctrl'             => true,
				'map_streetview_ctrl'       => true,
				'map_geo_ctrl'              => true,
				'map_traffic'               => false,
				'map_transit'               => false,
				'map_bicycling'             => false,
				'map_poi'                   => false,
				'map_marker_icon'           => '',
				'map_marker_icon_hover'     => '',
				'map_marker_height'         => 40,
				'map_marker_content'        => false,
				'map_marker_cluster'        => false,
				'map_cluster_size'          => 2,
				'map_cluster_radius'        => 50,
				'map_cluster_max_zoom'      => 14,
				'geo_type'                  => 'mapbox',
				'geo_countries'             => [],
				'geo_google_types'          => [],
				'geo_mapbox_types'          => [],
				'geo_placeholder'           => __( 'Enter a location', 'wpgb-map-facet' ),
				'geo_locate_me'             => true,
				'auto_geo_locate'           => false,
				'geo_locate_me_label'       => __( 'Locate Me', 'wpgb-map-facet' ),
				'geo_location_circle'       => true,
				'geo_circle_color'          => 'rgb(54, 138, 252)',
				'geo_radius_control'        => true,
				'geo_radius_label'          => __( 'Show results within', 'wpgb-map-facet' ),
				'geo_radius_unit'           => 'km',
				'geo_radius_def'            => 25,
				'geo_radius_min'            => 1,
				'geo_radius_max'            => 150,
			]
		);
	}

	/**
	 * Save post/term/user map fields
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array   $fields Holds field values to save.
	 * @param string  $type   Object type (post/term/user).
	 * @param integer $id     Object id.
	 * @return array
	 */
	public function save_fields( $fields, $type = '', $id = 0 ) {

		if ( ! isset( $fields['map_lng'], $fields['map_lat'] ) ) {
			return $fields;
		}

		$coordinates = [
			'lng' => $fields['map_lng'],
			'lat' => $fields['map_lat'],
		];

		update_metadata( $type, $id, '_wpgb_map_coordinates', $coordinates );
		$this->trigger_indexer( $type, $id );

		return $fields;

	}

	/**
	 * Manually trigger indexer on save in preview mode (grid settings)
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param string  $type Object type (post/term/user).
	 * @param integer $id   Object id.
	 * @return array
	 */
	public function trigger_indexer( $type, $id ) {

		// For preview mode ajax popup only.
		if ( ! wp_doing_ajax() && ! ( defined( 'REST_REQUEST' ) && REST_REQUEST ) ) {
			return;
		}

		switch ( $type ) {
			case 'post':
				do_action( 'save_post', $id, get_post( $id ), true );
				break;
			case 'user':
				do_action( 'profile_update', $id, get_user_by( 'id', $id ) );
				break;
			case 'term':
				$term = get_term( $id );
				do_action( 'edited_term', $id, $term->term_taxonomy_id, $term->taxonomy );
				break;
		}
	}
}
