<?php
/**
 * Map Facet
 *
 * @package   WP Grid Builder
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

$leaflet_styles    = [];
$leaflet_providers = include WPGB_MAP_PATH . 'includes/facets/providers.php';

foreach ( $leaflet_providers as $provider => $args ) {

	$leaflet_styles[] = [
		'value' => $provider,
		'label' => $provider,
	];

	if ( ! empty( $args['variants'] ) ) {

		foreach ( $args['variants'] as $variant => $atts ) {
			$leaflet_styles[] = [
				'value' => $provider . '.' . $variant,
				'label' => $provider . ' ' . $variant,
			];
		}
	}
}

usort(
	$leaflet_styles,
	function( $a, $b ) {
		return strnatcasecmp( $a['value'], $b['value'] );
	}
);

$mapbox_styles = array_merge(
	apply_filters(
		'wp_grid_builder_map/mapbox_styles',
		[
			'mapbox://styles/mapbox/streets-v11'          => __( 'Street', 'wpgb-map-facet' ),
			'mapbox://styles/mapbox/light-v10'            => __( 'Light', 'wpgb-map-facet' ),
			'mapbox://styles/mapbox/dark-v10'             => __( 'Dark', 'wpgb-map-facet' ),
			'mapbox://styles/mapbox/outdoors-v9'          => __( 'Outdoors', 'wpgb-map-facet' ),
			'mapbox://styles/mapbox/satellite-streets-v9' => __( 'Satellite', 'wpgb-map-facet' ),
		]
	),
	[ 'custom' => __( 'Custom', 'wpgb-map-facet' ) ]
);

$marker_styles = [
	'type'   => 'group',
	'fields' => [
		'marker_spacings'   => [
			'type'        => 'panel',
			'group'       => 'marker_style_panels',
			'title'       => __( 'Spacing', 'wpgb-map-facet' ),
			'className'   => 'wpgb-marker-style-popup-panel',
			'clearButton' => true,
			'fields'      => [
				'padding' => [
					'type'  => 'padding',
					'label' => _x( 'Padding', 'CSS padding', 'wpgb-map-facet' ),
				],
				'margin'  => [
					'type'  => 'margin',
					'label' => _x( 'Margin', 'CSS margin', 'wpgb-map-facet' ),
				],
			],
		],
		'marker_borders'    => [
			'type'        => 'panel',
			'group'       => 'marker_style_panels',
			'title'       => __( 'Borders', 'wpgb-map-facet' ),
			'className'   => 'wpgb-marker-style-popup-panel',
			'clearButton' => true,
			'fields'      => [
				'borders'       => [
					'type'         => 'border',
					'colorSchemes' => true,
				],
				'border-radius' => [
					'type'  => 'radius',
					'label' => __( 'Radius', 'wpgb-map-facet' ),
				],
			],
		],
		'marker_typography' => [
			'type'        => 'panel',
			'group'       => 'marker_style_panels',
			'title'       => __( 'Typography', 'wpgb-map-facet' ),
			'className'   => 'wpgb-marker-style-popup-panel',
			'clearButton' => true,
			'fields'      => [
				'typography' => [
					'type' => 'typography',
				],
			],
		],
		'marker_colors'     => [
			'type'        => 'panel',
			'group'       => 'marker_style_panels',
			'title'       => __( 'Colors', 'wpgb-map-facet' ),
			'className'   => 'wpgb-marker-style-popup-panel',
			'clearButton' => true,
			'fields'      => [
				'color'      => [
					'type'         => 'color',
					'label'        => __( 'Text', 'wpgb-map-facet' ),
					'colorSchemes' => true,
				],
				'background' => [
					'type'         => 'color',
					'label'        => __( 'Background', 'wpgb-map-facet' ),
					'gradient'     => true,
					'colorSchemes' => true,
				],
			],
		],
		'marker_shadows'    => [
			'type'        => 'panel',
			'group'       => 'marker_style_panels',
			'title'       => __( 'Shadows', 'wpgb-map-facet' ),
			'className'   => 'wpgb-marker-style-popup-panel',
			'clearButton' => true,
			'fields'      => [
				'box-shadows'  => [
					'type'  => 'box-shadow',
					'label' => __( 'Box Shadow', 'wpgb-map-facet' ),
				],
				'text-shadows' => [
					'type'  => 'text-shadow',
					'label' => __( 'Text Shadow', 'wpgb-map-facet' ),
				],
			],
		],
	],
];

return [
	[
		'type'   => 'clone',
		'fields' => [
			'content_type',
			'source',
			'include',
			'exclude',
			'parent',
			'child_of',
		],
	],
	[
		'type'   => 'fieldset',
		'legend' => __( 'Map', 'wpgb-map-facet' ),
		'fields' => [
			'map_control_style' => [
				'type'    => 'custom',
				'content' => '',
			],
			'map_type'          => [
				'type'    => 'button',
				'label'   => __( 'API Type', 'wpgb-map-facet' ),
				'options' => [
					[
						'value' => 'leaflet',
						'label' => 'Leaflet',
					],
					[
						'value' => 'mapbox',
						'label' => 'Mapbox',
					],
					[
						'value' => 'google',
						'label' => 'Google',
					],
				],
			],
			'map_access_token'  => [
				'type'         => 'password',
				'label'        => __( 'Access Token', 'wpgb-map-facet' ),
				'autoComplete' => 'new-password',
				'condition'    => [
					[
						'field'   => 'map_type',
						'compare' => '===',
						'value'   => 'mapbox',
					],
				],
			],
			'map_api_key'       => [
				'type'         => 'password',
				'label'        => __( 'API Key', 'wpgb-map-facet' ),
				'autoComplete' => 'new-password',
				'condition'    => [
					[
						'field'   => 'map_type',
						'compare' => '===',
						'value'   => 'google',
					],
				],
			],
			'map_limit_results' => [
				'type'        => 'select',
				'label'       => __( 'Map Results', 'wpgb-map-facet' ),
				'isClearable' => false,
				'options'     => [
					[
						'value' => 'all',
						'label' => __( 'Display all markers', 'wpgb-map-facet' ),
					],
					[
						'value' => 'page',
						'label' => __( 'Display markers in current page', 'wpgb-map-facet' ),
					],
				],
			],
			'map_appearance'    => [
				'type'   => 'grid',
				'fields' => [
					'map_google_style'     => [
						'type'         => 'select',
						'label'        => __( 'Map Style or ID', 'wpgb-map-facet' ),
						'isClearable'  => false,
						'isCreatable'  => true,
						'isSearchable' => true,
						'options'      => [
							[
								'value' => 'roadmap',
								'label' => __( 'Roadmap', 'wpgb-map-facet' ),
							],
							[
								'value' => 'satellite',
								'label' => __( 'Satellite', 'wpgb-map-facet' ),
							],
							[
								'value' => 'hybrid',
								'label' => __( 'Hybrid', 'wpgb-map-facet' ),
							],
							[
								'value' => 'terrain',
								'label' => __( 'Terrain', 'wpgb-map-facet' ),
							],
						],
						'condition'    => [
							[
								'field'   => 'map_type',
								'compare' => '===',
								'value'   => 'google',
							],
						],
					],
					'map_mapbox_style'     => [
						'type'        => 'select',
						'label'       => __( 'Map Style', 'wpgb-map-facet' ),
						'isClearable' => false,
						'options'     => array_map(
							function( $value, $label ) {

								return [
									'value' => $value,
									'label' => $label,
								];
							},
							array_keys( $mapbox_styles ),
							$mapbox_styles
						),
						'condition'   => [
							[
								'field'   => 'map_type',
								'compare' => '===',
								'value'   => 'mapbox',
							],
						],
					],
					'map_mapbox_style_url' => [
						'type'      => 'text',
						'label'     => __( 'Map Style URL', 'wpgb-map-facet' ),
						'condition' => [
							[
								'field'   => 'map_type',
								'compare' => '===',
								'value'   => 'mapbox',
							],
							[
								'field'   => 'map_mapbox_style',
								'compare' => '===',
								'value'   => 'custom',
							],
						],
					],
					'map_leaflet_style'    => [
						'type'         => 'select',
						'label'        => __( 'Map Style', 'wpgb-map-facet' ),
						'isClearable'  => false,
						'isSearchable' => true,
						'options'      => $leaflet_styles,
						'condition'    => [
							[
								'field'   => 'map_type',
								'compare' => '===',
								'value'   => 'leaflet',
							],
						],
					],
					'map_ratio'            => [
						'type'   => 'group',
						'label'  => __( 'Map Aspect Ratio', 'wpgb-map-facet' ),
						'fields' => [
							'row' => [
								'type'   => 'row',
								'fields' => [
									'x' => [
										'type'    => 'number',
										'tooltip' => 'X',
										'min'     => 1,
										'max'     => 999,
									],
									'y' => [
										'type'    => 'number',
										'tooltip' => 'Y',
										'min'     => 1,
										'max'     => 999,
									],
								],
							],
						],
					],
				],
			],
			'map_coordinates'   => [
				'type'   => 'popover',
				'label'  => __( 'Default Coordinates', 'wpgb-map-facet' ),
				'fields' => [
					[
						'type'   => 'row',
						'fields' => [
							'map_lat' => [
								'type'  => 'number',
								'label' => __( 'Latitude', 'wpgb-map-facet' ),
								'step'  => 0.000001,
								'min'   => -180,
								'max'   => 180,
							],
							'map_lng' => [
								'type'  => 'number',
								'label' => __( 'Longitude', 'wpgb-map-facet' ),
								'step'  => 0.000001,
								'min'   => -180,
								'max'   => 180,
							],
						],
					],
					[
						'type'      => 'row',
						'className' => 'wpgb-components-row__icon',
						'fields'    => [
							'map_zoom'     => [
								'type'  => 'number',
								'label' => __( 'Zoom', 'wpgb-map-facet' ),
								'icon'  => 'fullscreen',
								'min'   => 0,
								'max'   => 20,
							],
							'map_min_zoom' => [
								'type'  => 'number',
								'label' => (
									'<span class="wpgb-components-base-control__label-light">' .
										__( 'Min', 'wpgb-map-facet' ) .
									'</span>' .
									'<span class="wpgb-components-visually-hidden">' .
										__( 'Zoom', 'wpgb-map-facet' ) .
									'</span>'
								),
								'min'   => 0,
								'max'   => 20,
							],
							'map_max_zoom' => [
								'type'  => 'number',
								'label' => (
									'<span class="wpgb-components-base-control__label-light">' .
										__( 'Max', 'wpgb-map-facet' ) .
									'</span>' .
									'<span class="wpgb-components-visually-hidden">' .
										__( 'Zoom', 'wpgb-map-facet' ) .
									'</span>'
								),
								'min'   => 0,
								'max'   => 20,
							],
						],
					],
				],
			],
			'map_behaviour'     => [
				'type'   => 'popover',
				'label'  => __( 'Map Beaviour', 'wpgb-map-facet' ),
				'fields' => [
					'map_dragging'              => [
						'type'  => 'toggle',
						'label' => __( 'Drag to Pan', 'wpgb-map-facet' ),
					],
					'map_scrolling'             => [
						'type'      => 'toggle',
						'label'     => __( 'Scroll to Zoom', 'wpgb-map-facet' ),
						'condition' => [
							'relation' => 'OR',
							[
								'field'   => 'map_type',
								'compare' => '!==',
								'value'   => 'google',
							],
							[
								[
									'field'   => 'map_type',
									'compare' => '===',
									'value'   => 'google',
								],
								[
									'field'   => 'map_dragging',
									'compare' => '!=',
									'value'   => 0,
								],
							],
						],
					],
					'map_filtering'             => [
						'type'  => 'toggle',
						'label' => __( 'Pan to Filter', 'wpgb-map-facet' ),
					],
					'map_pan_search_ctrl'       => [
						'type'  => 'toggle',
						'label' => __( 'Filter Checkbox', 'wpgb-map-facet' ),
					],
					'map_pan_search_ctrl_label' => [
						'type'        => 'text',
						'label'       => __( 'Checkbox Label', 'wpgb-map-facet' ),
						'placeholder' => __( 'Search as I move the map', 'wpgb-map-facet' ),
						'condition'   => [
							[
								'field'   => 'map_pan_search_ctrl',
								'compare' => '==',
								'value'   => 1,
							],
						],
					],
				],
			],
			'map_controls'      => [
				'type'   => 'popover',
				'label'  => __( 'Map Controls', 'wpgb-map-facet' ),
				'fields' => [
					'map_scale_ctrl'      => [
						'type'      => 'toggle',
						'label'     => __( 'Scale', 'wpgb-map-facet' ),
						'condition' => [
							[
								'field'   => 'map_type',
								'compare' => 'IN',
								'value'   => [ 'google', 'mapbox' ],
							],
						],
					],
					'map_zoom_ctrl'       => [
						'type'  => 'toggle',
						'label' => __( 'Zoom', 'wpgb-map-facet' ),
					],
					'map_rotate_ctrl'     => [
						'type'      => 'toggle',
						'label'     => __( 'Rotate', 'wpgb-map-facet' ),
						'condition' => [
							[
								'field'   => 'map_type',
								'compare' => '===',
								'value'   => 'google',
							],
						],
					],
					'map_fullscreen_ctrl' => [
						'type'      => 'toggle',
						'label'     => __( 'Fullscreen', 'wpgb-map-facet' ),
						'condition' => [
							[
								'field'   => 'map_type',
								'compare' => 'IN',
								'value'   => [ 'google', 'mapbox' ],
							],
						],
					],
					'map_type_ctrl'       => [
						'type'      => 'toggle',
						'label'     => __( 'Map Type', 'wpgb-map-facet' ),
						'condition' => [
							[
								'field'   => 'map_type',
								'compare' => '===',
								'value'   => 'google',
							],
						],
					],
					'map_streetview_ctrl' => [
						'type'      => 'toggle',
						'label'     => __( 'Street View', 'wpgb-map-facet' ),
						'condition' => [
							[
								'field'   => 'map_type',
								'compare' => '===',
								'value'   => 'google',
							],
						],
					],
					'map_geo_ctrl'        => [
						'type'      => 'toggle',
						'label'     => __( 'Geolocate', 'wpgb-map-facet' ),
						'condition' => [
							[
								'field'   => 'map_type',
								'compare' => '===',
								'value'   => 'mapbox',
							],
						],
					],
				],
			],
			'map_layers'        => [
				'type'      => 'popover',
				'label'     => __( 'Map Layers', 'wpgb-map-facet' ),
				'fields'    => [
					'map_traffic'   => [
						'type'  => 'toggle',
						'label' => __( 'Traffic', 'wpgb-map-facet' ),
					],
					'map_transit'   => [
						'type'  => 'toggle',
						'label' => __( 'Transit', 'wpgb-map-facet' ),
					],
					'map_bicycling' => [
						'type'  => 'toggle',
						'label' => __( 'Bicycling', 'wpgb-map-facet' ),
					],
					'map_poi'       => [
						'type'  => 'toggle',
						'label' => __( 'Points of Interest', 'wpgb-map-facet' ),
					],
				],
				'condition' => [
					[
						'field'   => 'map_type',
						'compare' => '===',
						'value'   => 'google',
					],
					[
						'field'   => 'map_type',
						'compare' => '===',
						'value'   => 'google',
					],
				],
			],
		],
	],
	[
		'type'        => 'fieldset',
		'legend'      => __( 'Markers', 'wpgb-map-facet' ),
		'description' => __( 'Assign custom markers conditionally to the map. Conditions will be checked in the following order.', 'wpgb-map-facet' ),
		'fields'      => [
			'marker_icons'      => [
				'type'   => 'grid',
				'hidden' => true,
				'fields' => [
					'map_marker_icon'       => [
						'type'        => 'image',
						'layout'      => 'compact',
						'label'       => __( 'Marker Icon', 'wpgb-map-facet' ),
						'placeholder' => __( 'Select Marker', 'wpgb-map-facet' ),
					],
					'map_marker_icon_hover' => [
						'type'        => 'image',
						'layout'      => 'compact',
						'label'       => __( 'Marker Icon (Highlighted)', 'wpgb-map-facet' ),
						'placeholder' => __( 'Select Marker', 'wpgb-map-facet' ),
						'info'        => __( 'The highlighted icon is used instead of the default icon when hovering cards in a grid.', 'wpgb-map-facet' ),
					],
				],
			],
			'map_marker_height' => [
				'type'      => 'number',
				'hidden'    => true,
				'label'     => __( 'Marker Size (px)', 'wpgb-map-facet' ),
				'condition' => [
					[
						'field'   => 'map_marker_icon',
						'compare' => '!==',
						'value'   => '',
					],
				],
			],
			'map_markers'       => [
				'type'     => 'repeater',
				'addLabel' => __( 'Add Marker', 'wpgb-map-facet' ),
				'minRows'  => 0,
				'maxRows'  => 10,
				'fields'   => [
					'marker_type'          => [
						'type'    => 'button',
						'label'   => __( 'Type', 'wpgb-map-facet' ),
						'options' => [
							[
								'value' => 'image',
								'label' => __( 'Image', 'wpgb-map-facet' ),
							],
							[
								'value' => 'content',
								'label' => __( 'Content', 'wpgb-map-facet' ),
							],
						],
					],
					'marker_post_content'  => [
						'name'         => 'marker_content',
						'type'         => 'select',
						'label'        => __( 'Marker Content', 'wpgb-map-facet' ),
						'placeholder'  => __( 'Select a data', 'wpgb-map-facet' ),
						'isClearable'  => true,
						'isSearchable' => true,
						'options'      => [
							[
								'value' => 'custom',
								'label' => __( 'Custom Content', 'wpgb-map-facet' ),
							],
							[
								'value' => 'post_title',
								'label' => __( 'Post Title', 'wpgb-map-facet' ),
							],
							[
								'value' => 'post_author',
								'label' => __( 'Post Author', 'wpgb-map-facet' ),
							],
							[
								'value' => 'post_date',
								'label' => __( 'Post Date', 'wpgb-map-facet' ),
							],
							[
								'value' => 'post_status',
								'label' => __( 'Post Status', 'wpgb-map-facet' ),
							],
							[
								'value' => 'post_type',
								'label' => __( 'Post Type', 'wpgb-map-facet' ),
							],
							[
								'value' => 'post_format',
								'label' => __( 'Post Format', 'wpgb-map-facet' ),
							],
							[
								'value' => 'post_comments_number',
								'label' => __( 'Post Comments Count', 'wpgb-map-facet' ),
							],
							[
								'value' => 'post_metadata',
								'label' => __( 'Post Custom Field', 'wpgb-map-facet' ),
							],
						],
						'condition'    => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
							[
								'field'   => 'content_type',
								'compare' => '===',
								'value'   => 'post',
							],
						],
					],
					'marker_term_content'  => [
						'name'         => 'marker_content',
						'type'         => 'select',
						'label'        => __( 'Marker Content', 'wpgb-map-facet' ),
						'placeholder'  => __( 'Select a data', 'wpgb-map-facet' ),
						'isClearable'  => true,
						'isSearchable' => true,
						'options'      => [
							[
								'value' => 'custom',
								'label' => __( 'Custom Content', 'wpgb-map-facet' ),
							],
							[
								'value' => 'term_name',
								'label' => __( 'Term Name', 'wpgb-map-facet' ),
							],
							[
								'value' => 'term_taxonomy',
								'label' => __( 'Term Taxonomy', 'wpgb-map-facet' ),
							],
							[
								'value' => 'term_count',
								'label' => __( 'Term Posts Count', 'wpgb-map-facet' ),
							],
							[
								'value' => 'term_metadata',
								'label' => __( 'Term Custom Field', 'wpgb-map-facet' ),
							],
						],
						'condition'    => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
							[
								'field'   => 'content_type',
								'compare' => '===',
								'value'   => 'term',
							],
						],
					],
					'marker_user_content'  => [
						'name'         => 'marker_content',
						'type'         => 'select',
						'label'        => __( 'Marker Content', 'wpgb-map-facet' ),
						'placeholder'  => __( 'Select a data', 'wpgb-map-facet' ),
						'isClearable'  => true,
						'isSearchable' => true,
						'options'      => [
							[
								'value' => 'custom',
								'label' => __( 'Custom Content', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_login',
								'label' => __( 'Username', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_display_name',
								'label' => __( 'User Display Name', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_first_name',
								'label' => __( 'User First Name', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_last_name',
								'label' => __( 'User Last Name', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_nickname',
								'label' => __( 'User Nickname', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_email',
								'label' => __( 'User Email', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_url',
								'label' => __( 'User Website', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_roles',
								'label' => __( 'User Roles', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_post_count',
								'label' => __( 'User Posts Count', 'wpgb-map-facet' ),
							],
							[
								'value' => 'user_metadata',
								'label' => __( 'User Custom Field', 'wpgb-map-facet' ),
							],
						],
						'condition'    => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
							[
								'field'   => 'content_type',
								'compare' => '===',
								'value'   => 'user',
							],
						],
					],
					'marker_custom_text'   => [
						'type'      => 'text',
						'label'     => __( 'Custom Content', 'wpgb-map-facet' ),
						'condition' => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
							[
								'field'   => 'marker_content',
								'compare' => '===',
								'value'   => 'custom',
							],
						],
					],
					'marker_meta'          => [
						'type'        => 'select',
						'label'       => __( 'Custom Field', 'wpgb-map-facet' ),
						'placeholder' => __( 'Enter a field name', 'wpgb-map-facet' ),
						'async'       => [
							'wpgb/v2/metadata?object=registered',
							'wpgb/v2/metadata?object=post',
							'wpgb/v2/metadata?object=term',
							'wpgb/v2/metadata?object=user',
						],
						'condition'   => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
							[
								'relation' => 'OR',
								[
									[
										'field'   => 'content_type',
										'compare' => '===',
										'value'   => 'post',
									],
									[
										'field'   => 'marker_content',
										'compare' => '===',
										'value'   => 'post_metadata',
									],
								],
								[
									[
										'field'   => 'content_type',
										'compare' => '===',
										'value'   => 'term',
									],
									[
										'field'   => 'marker_content',
										'compare' => '===',
										'value'   => 'term_metadata',
									],
								],
								[
									[
										'field'   => 'content_type',
										'compare' => '===',
										'value'   => 'user',
									],
									[
										'field'   => 'marker_content',
										'compare' => '===',
										'value'   => 'user_metadata',
									],
								],
							],
						],
					],
					'marker_prefix_suffix' => [
						'type'      => 'grid',
						'fields'    => [
							'marker_prefix' => [
								'type'        => 'text',
								'label'       => __( 'Prefix', 'wpgb-map-facet' ),
								'whiteSpaces' => true,
								'angleQuotes' => true,
							],
							'marker_suffix' => [
								'type'        => 'text',
								'label'       => __( 'Suffix', 'wpgb-map-facet' ),
								'whiteSpaces' => true,
								'angleQuotes' => true,
							],
						],
						'condition' => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
							[
								'field'   => 'marker_content',
								'compare' => '!==',
								'value'   => 'custom',
							],
						],
					],
					'marker_meta_settings' => [
						'type'      => 'group',
						'fields'    => [
							'meta_type'    => [
								'type'    => 'button',
								'label'   => __( 'Field Type', 'wpgb-map-facet' ),
								'options' => [
									[
										'value' => 'text',
										'label' => __( 'Text', 'wpgb-map-facet' ),
									],
									[
										'value' => 'number',
										'label' => __( 'Number', 'wpgb-map-facet' ),
									],
									[
										'value' => 'date',
										'label' => __( 'Date', 'wpgb-map-facet' ),
									],
								],
							],
							'meta_date'    => [
								'type'      => 'grid',
								'fields'    => [
									'meta_input_date'  => [
										'type'        => 'text',
										'label'       => __( 'Input Date Format', 'wpgb-map-facet' ),
										'placeholder' => __( 'Auto (e.g: y-m-d)', 'wpgb-map-facet' ),
									],
									'meta_output_date' => [
										'type'        => 'text',
										'label'       => __( 'Output Date Format', 'wpgb-map-facet' ),
										'help'        => sprintf(
											// translators: 1: opening anchor tag 2: closing anchor tag.
											__( 'Learn more about %1$sPHP date format%2$s.', 'wpgb-map-facet' ),
											'<a href="https://www.php.net/manual/datetime.format.php" target="_blank">',
											'</a>'
										),
										'placeholder' => __( 'Default (e.g: F j, Y)', 'wpgb-map-facet' ),

									],
								],
								'condition' => [
									[
										'field'   => 'meta_type',
										'compare' => '===',
										'value'   => 'date',
									],
								],
							],
							'meta_decimal' => [
								'type'      => 'grid',
								'columns'   => 3,
								'fields'    => [
									'meta_decimal_places' => [
										'type'  => 'number',
										'label' => __( 'Decimal Places', 'wpgb-map-facet' ),
										'value' => 0,
									],
									'meta_decimal_separator' => [
										'type'         => 'text',
										'label'        => __( 'Decimal Separator', 'wpgb-map-facet' ),
										'defaultValue' => '.',
									],
									'meta_thousands_separator' => [
										'type'        => 'text',
										'label'       => __( 'Thousands Separator', 'wpgb-map-facet' ),
										'whiteSpaces' => true,
									],
								],
								'condition' => [
									[
										'field'   => 'meta_type',
										'compare' => '===',
										'value'   => 'number',
									],
								],
							],
						],
						'condition' => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
							[
								'field'   => 'marker_content',
								'compare' => 'IN',
								'value'   => [ 'post_metadata', 'term_metadata', 'user_metadata' ],
							],
						],
					],
					'marker_icons'         => [
						'type'      => 'grid',
						'columns'   => 3,
						'fields'    => [
							'marker_image_idle'  => [
								'type'        => 'image',
								'layout'      => 'compact',
								'label'       => __( 'Image', 'wpgb-map-facet' ),
								'placeholder' => __( 'Select Image', 'wpgb-map-facet' ),
							],
							'marker_image_hover' => [
								'type'        => 'image',
								'layout'      => 'compact',
								'label'       => __( 'Image (Highlighted)', 'wpgb-map-facet' ),
								'placeholder' => __( 'Select Image', 'wpgb-map-facet' ),
							],
							'marker_image_size'  => [
								'type'        => 'number',
								'label'       => __( 'Image Size', 'wpgb-map-facet' ),
								'suffix'      => 'px',
								'placeholder' => 40,
							],
						],
						'condition' => [
							[
								'field'   => 'marker_type',
								'compare' => '!==',
								'value'   => 'content',
							],
						],
					],
					'marker_idle_styles'   => [
						'type'      => 'popover',
						'label'     => __( 'Style', 'wpgb-map-facet' ),
						'fields'    => [
							'marker_idle_style' => $marker_styles,
						],
						'condition' => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
						],
					],
					'marker_hover_styles'  => [
						'type'      => 'popover',
						'label'     => __( 'Style (Highlighted)', 'wpgb-map-facet' ),
						'fields'    => [
							'marker_highlighted_style' => $marker_styles,
						],
						'condition' => [
							[
								'field'   => 'marker_type',
								'compare' => '===',
								'value'   => 'content',
							],
						],
					],
					'conditions'           => [
						'type'    => 'condition',
						'label'   => __( 'Conditions', 'wpgb-map-facet' ),
						'context' => 'map',
					],
				],
			],
			'map_markers_style' => [
				'type'   => 'text',
				'hidden' => true,
			],
			'google'            => [
				'type'   => 'fonts',
				'hidden' => true,
			],
		],
	],
	[
		'type'        => 'fieldset',
		'legend'      => __( 'Popups', 'wpgb-map-facet' ),
		'description' => __( 'Assign cards conditionally to marker popups in the map. Conditions will be checked in the following order.', 'wpgb-map-facet' ),
		'fields'      => [
			'map_marker_content' => [
				'type'  => 'toggle',
				'label' => __( 'Open Popup on click', 'wpgb-map-facet' ),
			],
			'map_popups'         => [
				'type'      => 'repeater',
				'label'     => __( 'Cards', 'wpgb-map-facet' ),
				'addLabel'  => __( 'Add Card', 'wpgb-map-facet' ),
				'minRows'   => 0,
				'maxRows'   => 10,
				'fields'    => [
					'card'       => [
						'type'        => 'select',
						'label'       => __( 'Card', 'wpgb-map-facet' ),
						'placeholder' => __( 'Default Card', 'wpgb-map-facet' ),
						'async'       => 'wpgb/v2/objects?object=cards&orderby=name&order=asc&fields=id,name',
					],
					'attributes' => [
						'type'   => 'group',
						'fields' => [
							'width' => [
								'type'        => 'number',
								'label'       => __( 'Max Width', 'wpgb-map-facet' ),
								'placeholder' => __( '300', 'wpgb-map-facet' ),
								'suffix'      => 'px',
							],
						],
					],
					'conditions' => [
						'type'    => 'condition',
						'label'   => __( 'Conditions', 'wpgb-map-facet' ),
						'context' => 'map',
					],
				],
				'condition' => [
					[
						'field'   => 'map_marker_content',
						'compare' => '==',
						'value'   => 1,
					],
				],
			],
		],
	],
	[
		'type'   => 'fieldset',
		'legend' => __( 'Clusters', 'wpgb-map-facet' ),
		'fields' => [
			'map_marker_cluster' => [
				'type'  => 'toggle',
				'label' => __( 'Marker Clustering', 'wpgb-map-facet' ),
			],
			'grid'               => [
				'type'    => 'grid',
				'columns' => 3,
				'fields'  => [
					'map_cluster_size'     => [
						'type'      => 'number',
						'label'     => __( 'Min Markers in Cluster', 'wpgb-map-facet' ),
						'help'      => __( 'Minimum markers to form a cluster.', 'wpgb-map-facet' ),
						'min'       => 2,
						'max'       => 999999,
						'step'      => 1,
						'condition' => [
							[
								'field'   => 'map_marker_cluster',
								'compare' => '==',
								'value'   => 1,
							],
							[
								'field'   => 'map_type',
								'compare' => 'IN',
								'value'   => [ 'google', 'mapbox' ],
							],
						],
					],
					'map_cluster_radius'   => [
						'type'      => 'number',
						'label'     => __( 'Cluster Radius', 'wpgb-map-facet' ),
						'help'      => __( 'Cluster radius when clustering points.', 'wpgb-map-facet' ),
						'min'       => 1,
						'max'       => 999999,
						'step'      => 1,
						'suffix'    => 'px',
						'condition' => [
							[
								'field'   => 'map_marker_cluster',
								'compare' => '==',
								'value'   => 1,
							],
						],
					],
					'map_cluster_max_zoom' => [
						'type'      => 'number',
						'label'     => __( 'Cluster Max Zoom', 'wpgb-map-facet' ),
						'help'      => __( 'Maximum zoom to cluster markers on.', 'wpgb-map-facet' ),
						'min'       => 1,
						'max'       => 20,
						'step'      => 1,
						'condition' => [
							[
								'field'   => 'map_marker_cluster',
								'compare' => '==',
								'value'   => 1,
							],
						],
					],
				],
			],
			'map_clusters'       => [
				'type'      => 'repeater',
				'label'     => __( 'Cluster Icons', 'wpgb-map-facet' ),
				'addLabel'  => __( 'Add Cluster', 'wpgb-map-facet' ),
				'rowLabel'  => '',
				'minRows'   => 0,
				'maxRows'   => 10,
				'fields'    => [
					'cluster_icon' => [
						'type'    => 'grid',
						'columns' => 2,
						'fields'  => [
							'cluster_count' => [
								'type'        => 'number',
								'label'       => __( 'Max Markers', 'wpgb-map-facet' ),
								'min'         => 1,
								'max'         => 999999,
								'placeholder' => 'infinity',
							],
							'cluster_size'  => [
								'type'        => 'number',
								'label'       => __( 'Cluster Size', 'wpgb-map-facet' ),
								'suffix'      => 'px',
								'placeholder' => 54,
							],
							'cluster_icon'  => [
								'type'        => 'image',
								'layout'      => 'compact',
								'label'       => __( 'Cluster Icon', 'wpgb-map-facet' ),
								'placeholder' => __( 'Select Image', 'wpgb-map-facet' ),
							],
							'cluster_color' => [
								'type'  => 'color',
								'label' => __( 'Cluster Color', 'wpgb-map-facet' ),
							],
						],
					],
				],
				'condition' => [
					[
						'field'   => 'map_marker_cluster',
						'compare' => '==',
						'value'   => 1,
					],
				],
			],
		],
	],
];
