<?php
/**
 * Geolocation facet
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes\Facets;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Geolocation Facet class
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Facets\Geo
 * @since 1.1.0
 */
class Geo extends Helpers {

	/**
	 * Constructor
	 *
	 * @since 1.1.0
	 * @access public
	 */
	public function __construct() {

		add_filter( 'wp_grid_builder/facet/response', [ $this, 'filter_response' ], 10, 3 );
		add_filter( 'wp_grid_builder/facet/weight', [ $this, 'facet_weight' ], 10, 2 );

	}

	/**
	 * Filter facet response to set locale
	 *
	 * @since 1.1.0
	 * @access public
	 *
	 * @param array $response Holds facet response.
	 * @param array $facet    Holds facet settings.
	 * @param array $items    Holds facet items.
	 * @return array
	 */
	public function filter_response( $response, $facet, $items ) {

		// Skip other facets or if already set.
		if ( 'geolocation' !== $facet['type'] || isset( $response['settings']['locale'] ) ) {
			return $response;
		}

		$response['settings']['locale'] = get_locale();

		return $response;

	}

	/**
	 * Set max weight to geolocation results to order by distances (post__in/include clauses)
	 *
	 * @since 1.1.3
	 * @access public
	 *
	 * @param array $weight Facet weight.
	 * @param array $facet  Holds facet settings.
	 * @return integer
	 */
	public function facet_weight( $weight, $facet ) {

		if ( 'geolocation' === $facet['type'] && ! empty( $facet['selected'] ) ) {
			$weight = PHP_INT_MAX - 9;
		}

		return $weight;

	}

	/**
	 * Render facet input
	 *
	 * @since 1.1.0
	 * @access public
	 *
	 * @param array $facet Holds facet settings.
	 * @param array $items Holds facet items.
	 * @return string Facet markup.
	 */
	public function render_facet( $facet, $items ) {

		$settings = $this->normalize( $facet['settings'] );
		$label    = $facet['title'] ?: __( 'Geolocation', 'wpgb-map-facet' );
		$output   = sprintf(
			'<div class="wpgb-geolocation-facet">
				<div class="wpgb-geolocation-input">
					<label>
						<span class="wpgb-sr-only">%1$s</span>
						<input class="wpgb-input" type="search" placeholder="%2$s" autocomplete="off">
						%3$s
					</label>
					%4$s
				</div>
				%5$s
			</div>',
			esc_html( $label ),
			esc_attr( $settings['geo_placeholder'] ),
			$this->search_icon(),
			$this->locate_button( $settings ),
			$this->radius_control( $settings )
		);

		return apply_filters( 'wp_grid_builder/facet/geolocation', $output, $facet );

	}

	/**
	 * Search icon
	 *
	 * @since 1.1.0
	 * @access public
	 *
	 * @return string
	 */
	public function search_icon() {

		$output  = '<svg class="wpgb-input-icon" viewBox="0 0 24 24" height="16" width="16" aria-hidden="true" focusable="false">';
		$output .= '<path d="M17.5 17.5 23 23Zm-16-7a9 9 0 1 1 9 9 9 9 0 0 1-9-9Z"/>';
		$output .= '</svg>';

		return $output;

	}

	/**
	 * Locate button
	 *
	 * @since 1.1.0
	 * @access public
	 *
	 * @param array $settings Holds facet settings.
	 * @return string
	 */
	public function locate_button( $settings ) {

		if ( empty( $settings['geo_locate_me'] ) ) {
			return '';
		}

		$output  = '<button type="button" class="wpgb-locate-button" hidden>';
		$output .= '<span class="wpgb-sr-only">' . esc_html( $settings['geo_locate_me_label'] ) . '</span>';
		$output .= '<svg viewBox="0 0 24 24" width="24" height="24" aria-hidden="true" focusable="false">';
		$output .= '<path d="M12 3a9 9 0 1 1-9 9 9 9 0 0 1 9-9ZM12 1v4m11 7h-4m-7 11v-4M1 12h4" stroke-linecap="square"/>';
		$output .= '<path d="M12 11a1 1 0 1 1-1 1 1 1 0 0 1 1-1Z"/>';
		$output .= '</svg>';
		$output .= '</button>';

		return $output;

	}

	/**
	 * Radius input control
	 *
	 * @since 1.1.0
	 * @access public
	 *
	 * @param array $settings Holds facet settings.
	 * @return string
	 */
	public function radius_control( $settings ) {

		if ( empty( $settings['geo_radius_control'] ) ) {
			return '';
		}

		return sprintf(
			'<label class="wpgb-geo-radius">%1$s&nbsp;<input type="number" value="%2$g" min="%3$g" max="%4$g">&nbsp;%5$s</label>',
			esc_html( $settings['geo_radius_label'] ),
			esc_attr( $settings['geo_radius_def'] ),
			esc_attr( $settings['geo_radius_min'] ),
			esc_attr( $settings['geo_radius_max'] ),
			esc_html( $settings['geo_radius_unit'] )
		);
	}

	/**
	 * Normalize facet settings
	 *
	 * @since 1.1.0
	 * @access public
	 *
	 * @param array $settings Holds facet settings.
	 * @return array
	 */
	public function normalize( $settings ) {

		return wp_parse_args(
			$settings,
			[
				'geo_location_circle' => true,
				'geo_circle_color'    => 'rgb(54, 138, 252)',
				'geo_placeholder'     => __( 'Enter a location', 'wpgb-map-facet' ),
				'geo_locate_me'       => true,
				'geo_locate_me_label' => __( 'Locate Me', 'wpgb-map-facet' ),
				'geo_radius_control'  => true,
				'geo_radius_label'    => __( 'Show results within', 'wpgb-map-facet' ),
				'geo_radius_unit'     => 'km',
				'geo_radius_def'      => 25,
				'geo_radius_min'      => 1,
				'geo_radius_max'      => 150,
			]
		);
	}

	/**
	 * Query object ids (post, user, term) for selected facet values
	 *
	 * @since 1.1.0
	 * @access public
	 *
	 * @param array $facet      Holds facet settings.
	 * @param array $query_vars Holds query vars.
	 * @return array Holds queried facet object ids.
	 */
	public function query_objects( $facet, $query_vars = [] ) {

		global $wpdb;

		if ( count( $facet['selected'] ) < 3 ) {
			return;
		}

		// To allow ASC/DESC order from post__in/include clauses.
		$order = isset( $query_vars['order'] ) && 'DESC' === strtoupper( $query_vars['order'] ) ? 'DESC' : 'ASC';
		$unit  = ! empty( $facet['geo_radius_unit'] ) ? $facet['geo_radius_unit'] : 'km';

		return $wpdb->get_col(
			$wpdb->prepare(
				"SELECT DISTINCT object_id, ( %f * acos(
					greatest( -1, least( 1, (
						cos( radians( %f ) ) *
						cos( radians( facet_value ) ) *
						cos( radians( facet_name ) - radians( %f ) ) +
						sin( radians( %f ) ) *
						sin( radians( facet_value ) )
					) ) )
				) ) AS distance
				FROM {$wpdb->prefix}wpgb_index
				WHERE slug = %s
				HAVING distance < %f
				ORDER BY distance $order",    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				'mi' === $unit ? 3959 : 6371, // Earth radius.
				$facet['selected'][0],        // Latitude.
				$facet['selected'][1],        // Longitude.
				$facet['selected'][0],        // Latitude.
				$facet['slug'],
				$facet['selected'][2]         // Radius.
			)
		);
	}
}
