<?php
/**
 * Facet Helpers
 *
 * @package   WP Grid Builder - Map Facet
 * @author    Loïc Blascos
 * @copyright 2019-2025 Loïc Blascos
 */

namespace WP_Grid_Builder_Map_Facet\Includes\Facets;

use WP_Grid_Builder_Map_Facet\Includes;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Facet Helpers
 *
 * @class WP_Grid_Builder_Map_Facet\Includes\Facets\Helpers
 * @since 1.0.0
 */
class Helpers {

	use Includes\Data;

	/**
	 * Image sources
	 *
	 * @since 2.0.0
	 * @access public
	 * @var array
	 */
	public $sources;

	/**
	 * Create geoJSON array from markers
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param array $markers Holds map markers.
	 * @param array $facet   Holds facet settings.
	 * @return array
	 */
	public function to_geojson( $markers, $facet ) {

		$icon = $this->get_default_icon( $facet );
		$icon = apply_filters( 'wp_grid_builder_map/marker_icon', $icon, $facet );
		$json = [
			'type'     => 'FeatureCollection',
			'features' => array_map(
				function( $marker ) use ( $icon, $facet ) {

					return [
						'type'       => 'Feature',
						'properties' => [
							'id'   => (int) $marker->id,
							'icon' => $this->get_feature_icon( $marker->id, $facet, $icon ),
						],
						'geometry'   => [
							'type'        => 'Point',
							'coordinates' => [
								(float) $marker->lng,
								(float) $marker->lat,
							],
						],
					];

				},
				$markers
			),
		];

		return apply_filters( 'wp_grid_builder_map/geojson', $json, $facet );

	}

	/**
	 * Get default icon
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param array $facet Holds facet settings.
	 * @return mixed
	 */
	public function get_default_icon( $facet ) {

		$settings = array_filter( $facet['settings'] );

		if ( ! empty( $settings['map_marker_icon'] ) ) {
			$settings['map_marker_icon'] = $this->get_feature_src( $settings['map_marker_icon'], 'medium' ) ?: '';
		}

		if ( ! empty( $settings['map_marker_icon_hover'] ) ) {
			$settings['map_marker_icon_hover'] = $this->get_feature_src( $settings['map_marker_icon_hover'], 'medium' ) ?: '';
		}

		return [
			'url'      => $settings['map_marker_icon'] ?? WPGB_MAP_URL . 'assets/imgs/marker.svg',
			'urlHover' => $settings['map_marker_icon_hover'] ?? '',
			'size'     => [
				'width'  => (int) ( $settings['map_marker_height'] ?? 40 ),
				'height' => (int) ( $settings['map_marker_height'] ?? 40 ),
			],
		];
	}

	/**
	 * Get feature icon
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param integer $object_id Marker object ID.
	 * @param array   $facet     Holds facet settings.
	 * @param array   $icon      Holds default icon.
	 * @return mixed
	 */
	public function get_feature_icon( $object_id, $facet, $icon ) {

		global $wpgb_marker_id;

		if ( empty( $facet['settings']['map_markers'] ) ) {
			return $icon;
		}

		$style_id = 0;
		$current  = [];
		$markers  = $facet['settings']['map_markers'];

		foreach ( (array) $markers as $index => $marker ) {

			$wpgb_marker_id = $object_id;

			if (
				// If no conditions, assign marker to any content type.
				empty( $marker['conditions'] ) ||
				apply_filters( 'wp_grid_builder/do_conditions', true, $marker['conditions'], 'marker', (int) $object_id )
			) {

				$style_id = ! empty( $marker['marker_idle_style'] ) ? $index + 1 : 0;
				$current  = $marker;
				break;

			}
		}

		return [
			'style_id' => $style_id,
			'content'  => $this->get_feature_content( $object_id, $facet, $current ),
			'urlHover' => $this->get_feature_src( $current['marker_image_hover'] ?? '', 'medium' ) ?: $icon['urlHover'],
			'url'      => $this->get_feature_src( $current['marker_image_idle'] ?? '', 'medium' ) ?: $icon['url'],
			'size'     => [
				'width'  => $current['marker_image_size'] ?? $icon['size']['width'],
				'height' => $current['marker_image_size'] ?? $icon['size']['height'],
			],
		];
	}

	/**
	 * Get feature icon
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param integer $id Image id to retrieve.
	 * @return mixed
	 */
	public function get_feature_src( $id ) {

		if ( isset( $this->sources[ $id ] ) ) {
			return $this->sources[ $id ];
		}

		$this->sources[ $id ] = esc_url( wp_get_attachment_image_url( $id, 'medium' ) );

		return $this->sources[ $id ];

	}

	/**
	 * Get feature content
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param integer $object_id Marker object ID.
	 * @param array   $facet     Holds facet settings.
	 * @param array   $args      Holds content arguments.
	 * @return string
	 */
	public function get_feature_content( $object_id, $facet, $args ) {

		if (
			empty( $args['marker_type'] ) ||
			empty( $args['marker_content'] ) ||
			'content' !== $args['marker_type']
		) {
			return '';
		}

		if ( 'custom' === $args['marker_content'] ) {
			return esc_html( $args['marker_custom_text'] ?? '' );
		}

		$args['field'] = $args['marker_meta'] ?? '';

		switch ( $facet['settings']['source'] ) {
			case 'user':
				$content = $this->user_data( '', $args['marker_content'], $object_id, $args );
				break;
			case 'term':
				$content = $this->term_data( '', $args['marker_content'], $object_id, $args );
				break;
			default:
				$content = $this->post_data( '', $args['marker_content'], $object_id, $args );
		}

		if ( '' === $content || null === $content ) {
			return '';
		}

		$settings = wpgb_normalize_metadata_settings( $args['marker_meta_settings'] ?? [] );

		return wp_kses_post(
			( $args['marker_prefix'] ?? '' ) .
			wpgb_format_metadata( $settings, $content ) .
			( $args['marker_suffix'] ?? '' )
		);
	}

	/**
	 * Get Leaflet provider
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param array $facet Holds facet settings.
	 * @return mixed
	 */
	public function get_clusters( $facet ) {

		$defaults = [
			[
				'cluster_count' => 9,
				'cluster_size'  => 54,
				'cluster_icon'  => esc_url( WPGB_MAP_URL . 'assets/imgs/cluster-1.svg' ),
				'cluster_color' => '',
			],
			[
				'cluster_count' => 99,
				'cluster_size'  => 56,
				'cluster_icon'  => esc_url( WPGB_MAP_URL . 'assets/imgs/cluster-2.svg' ),
				'cluster_color' => '',
			],
			[
				'cluster_count' => 999,
				'cluster_size'  => 66,
				'cluster_icon'  => esc_url( WPGB_MAP_URL . 'assets/imgs/cluster-3.svg' ),
				'cluster_color' => '',
			],
			[
				'cluster_count' => 9999,
				'cluster_size'  => 78,
				'cluster_icon'  => esc_url( WPGB_MAP_URL . 'assets/imgs/cluster-4.svg' ),
				'cluster_color' => '',
			],
			[
				'cluster_count' => 99999,
				'cluster_size'  => 90,
				'cluster_icon'  => esc_url( WPGB_MAP_URL . 'assets/imgs/cluster-5.svg' ),
				'cluster_color' => '',
			],
		];

		if ( empty( $facet['map_clusters'] ) ) {
			return $defaults;
		}

		$facet['map_clusters'] = array_filter(
			$facet['map_clusters'],
			function( $cluster ) {
				return ! empty( $cluster['cluster_icon'] );
			}
		);

		if ( empty( $facet['map_clusters'] ) ) {
			return $defaults;
		}

		return array_map(
			function( $cluster ) {

				$cluster['cluster_icon'] = $this->get_feature_src( $cluster['cluster_icon'] );

				return $cluster;

			},
			$facet['map_clusters']
		);
	}

	/**
	 * Get Leaflet provider
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $facet Holds facet settings.
	 * @return mixed
	 */
	public function get_provider( $facet ) {

		if ( ! empty( $facet['settings']['map_type'] ) && empty( $facet['settings']['map_leaflet_style'] ) ) {
			return '';
		}

		$providers = include WPGB_MAP_PATH . 'includes/facets/providers.php';
		$map_style = explode( '.', $facet['settings']['map_leaflet_style'] ?? '' );
		$map_style = array_pad( $map_style, 2, '' );
		$provider  = ! empty( $providers[ $map_style[0] ] ) ? $map_style[0] : 'Wikimedia';
		$provider  = $this->get_variant( $providers[ $provider ], $map_style[1] );
		$provider  = $this->get_attribution( $providers, $provider );

		unset( $provider['variants'] );

		return $provider;

	}

	/**
	 * Get Leaflet provider variant
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array  $provider Provider settings.
	 * @param string $variant  Provider variant name.
	 * @return array
	 */
	public function get_variant( $provider, $variant ) {

		if ( empty( $provider['variants'][ $variant ] ) ) {
			return $provider;
		}

		$variant = $provider['variants'][ $variant ];

		if ( is_string( $variant ) ) {
			$variant = [ 'options' => [ 'variant' => $variant ] ];
		}

		if ( ! empty( $variant['url'] ) ) {
			$provider['url'] = $variant['url'];
		}

		$provider['options'] = wp_parse_args( $variant['options'], $provider['options'] );

		return $provider;

	}

	/**
	 * Get Leaflet provider variant
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $providers Holds all providers.
	 * @param array $provider  Provider settings.
	 * @return array
	 */
	public function get_attribution( $providers, $provider ) {

		if ( empty( $provider['options']['attribution'] ) ) {
			return $provider;
		}

		$provider['options']['attribution'] = preg_replace_callback(
			'/{attribution.(\w*)\}/',
			function( $matches ) use ( $providers ) {

				if ( isset( $providers[ $matches[1] ] ) ) {
					return $providers[ $matches[1] ]['options']['attribution'];
				}

			},
			$provider['options']['attribution']
		);

		return $provider;

	}

	/**
	 * Get facet source type
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $facet Holds facet settings.
	 * @return mixed
	 */
	public function get_source_type( $facet ) {

		$source = 'post_type';

		if ( ! empty( $facet['field_type'] ) && 'post' !== $facet['field_type'] ) {
			$source = $facet['field_type'];
		}

		return $source;

	}

	/**
	 * Get facet fonts URL.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param array $facet Holds facet settings.
	 * @return string
	 */
	public function get_fonts_url( $facet ) {

		$google = $facet['settings']['map_fonts']['google'] ?? [];

		if ( empty( $google ) ) {
			return '';
		}

		$families = array_map( 'array_unique', $google );

		if ( empty( $families ) ) {
			return [];
		}

		array_walk(
			$families,
			function( $val, $key ) use ( &$query ) {
				$query[] = $key . ':' . implode( ',', $val );
			}
		);

		$query = implode( '|', $query );

		return add_query_arg(
			[ 'family' => rawurlencode( $query ) ],
			! empty( wpgb_get_option( 'bunny_fonts' ) ) ? 'https://fonts.bunny.net/css' : 'https://fonts.googleapis.com/css'
		);
	}
}
